{"vars":[{"containerName":"","name":"base","kind":2,"line":192},{"detail":"($class,@args)","signature":{"label":"new($class,@args)","parameters":[{"label":"$class"},{"label":"@args"}],"documentation":"1;\n# $Id: ProtMatrix.pm 16123 2009-09-17 12:57:27Z cjfields $\n#---------------------------------------------------------\n\n=head1 NAME\n\nBio::Matrix::PSM::ProtMatrix - SiteMatrixI implementation, holds a\nposition scoring matrix (or position weight matrix) with log-odds scoring\ninformation.\n\n=head1 SYNOPSIS\n\n   use Bio::Matrix::PSM::ProtMatrix;\n   # Create from memory by supplying probability matrix hash both as strings or\n   # arrays where the frequencies   Hash entries of the form lN refer to an array\n   # of position-specific log-odds scores for amino acid N. Hash entries of the\n   # form pN represent the position-specific probability of finding amino acid N.\n\n   my %param = (\n             'id' => 'A. thaliana protein atp1',\n             '-e_val' => $score,\n             'lS' => [ '-2', '3', '-3', '2', '-3', '1', '1', '3' ],\n             'lF' => [ '-1', '-4', '0', '-5', '0', '-5', '-4', '-4' ],\n             'lT' => [ '-1', '1', '0', '1', '-2', '-1', '0', '1' ],\n             'lN' => [ '-3', '-1', '-2', '3', '-5', '5', '-2', '0' ],\n             'lK' => [ '-2', '0', '-3', '2', '-3', '2', '-3', '-1' ],\n             'lY' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-4', '-4' ],\n             'lE' => [ '-3', '4', '-3', '2', '-4', '-2', '-3', '2' ],\n             'lV' => [ '0', '-2', '1', '-4', '1', '-4', '-1', '-3' ],\n             'lQ' => [ '-1', '0', '-2', '3', '-4', '1', '-3', '0' ],\n             'lM' => [ '8', '-3', '8', '-3', '1', '-3', '-3', '-3' ],\n             'lC' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-3', '-3' ],\n             'lL' => [ '1', '-3', '1', '-4', '3', '-4', '-2', '-4' ],\n             'lA' => [ '-2', '1', '-2', '0', '-2', '-2', '2', '2' ],\n             'lW' => [ '-2', '-4', '-3', '-5', '-4', '-5', '-5', '-5' ],\n             'lP' => [ '-3', '-2', '-4', '-3', '-1', '-3', '6', '-3' ],\n             'lH' => [ '-2', '-2', '-3', '-2', '-5', '-2', '-2', '-3' ],\n             'lD' => [ '-4', '-1', '-3', '1', '-3', '-1', '-3', '4' ],\n             'lR' => [ '-2', '-1', '-3', '0', '-4', '4', '-4', '-3' ],\n             'lI' => [ '0', '-3', '0', '-4', '6', '-4', '-2', '-2' ],\n             'lG' => [ '-4', '-2', '-4', '-2', '-5', '-3', '-1', '-2' ],\n             'pS' => [ '0', '33', '0', '16', '1', '12', '11', '25' ],\n             'pF' => [ '0', '0', '2', '0', '3', '0', '0', '0' ],\n             'pT' => [ '0', '8', '7', '10', '1', '2', '7', '8' ],\n             'pN' => [ '0', '0', '2', '13', '0', '36', '1', '4' ],\n             'pK' => [ '0', '5', '0', '13', '1', '15', '0', '2' ],\n             'pY' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pE' => [ '0', '41', '1', '12', '0', '0', '0', '15' ],\n             'pV' => [ '0', '3', '9', '0', '2', '0', '3', '1' ],\n             'pQ' => [ '0', '0', '0', '15', '0', '4', '0', '3' ],\n             'pM' => [ '100', '0', '66', '0', '2', '0', '0', '0' ],\n             'pC' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pL' => [ '0', '0', '8', '0', '25', '0', '4', '0' ],\n             'pA' => [ '0', '10', '1', '9', '2', '0', '22', '16' ],\n             'pW' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pP' => [ '0', '0', '0', '0', '3', '1', '45', '0' ],\n             'pH' => [ '0', '0', '0', '0', '0', '0', '1', '0' ],\n             'pD' => [ '0', '0', '1', '7', '2', '2', '0', '22' ],\n             'pR' => [ '0', '0', '0', '3', '0', '27', '0', '0' ],\n             'pI' => [ '0', '0', '3', '0', '59', '1', '2', '3' ],\n             'pG' => [ '0', '0', '0', '1', '0', '0', '4', '1' ],\n   );\n\n   my $matrix = Bio::Matrix::PSM::ProtMatrix( %param );\n\n\n   my $site = Bio::Matrix::PSM::ProtMatrix->new(%param);\n   # Or get it from a file:\n   use Bio::Matrix::PSM::IO;\n   my $psmIO = Bio::Matrix::PSM::IO->new(-file => $file, -format => 'psi-blast');\n   while (my $psm = $psmIO->next_psm) {\n      #Now we have a Bio::Matrix::PSM::Psm object, \n      # see Bio::Matrix::PSM::PsmI for details\n      #This is a Bio::Matrix::PSM::ProtMatrix object now\n      my $matrix = $psm->matrix;   \n   }\n\n   # Get a simple consensus, where alphabet is:\n   # {A, R, N, D, C, Q, E, G, H, I, L, K, M, F, P, S, T, W, Y, V,}\n   # choosing the highest probability or N if prob is too low\n   my $consensus = $site->consensus;\n\n   # Retrieving and using regular expressions:\n   my $regexp = $site->regexp;\n   my $count = grep($regexp,$seq);\n   my $count = ($seq=~ s/$regexp/$1/eg);\n   print \"Motif $mid is present $count times in this sequence\\n\";\n\n=head1 DESCRIPTION\n\nProtMatrix is designed to provide some basic methods when working with\nposition scoring (weight) matrices related to protein sequences.  A\nprotein PSM consists of 20 vectors with 20 frequencies (one per amino\nacid per position).  This is the minimum information you should\nprovide to construct a PSM object.  The vectors can be provided as\nstrings with frequencies where the frequency is {0..a} and a=1. This\nis the way MEME compressed representation of a matrix and it is quite\nuseful when working with relational DB.  If arrays are provided as an\ninput (references to arrays actually) they can be any number, real or\ninteger (frequency or count).\n\nWhen creating the object the constructor will check for positions that\nequal 0.  If such is found it will increase the count for all\npositions by one and recalculate the frequency.  Potential bug - if\nyou are using frequencies and one of the positions is 0 it will change\nsignificantly.  However, you should never have frequency that equals\n0.\n\nThrows an exception if: You mix as an input array and string (for\nexample A matrix is given as array, C - as string).  The position\nvector is (0,0,0,0).  One of the probability vectors is shorter than\nthe rest.\n\nSummary of the methods I use most frequently (details bellow):\n\n   iupac - return IUPAC compliant consensus as a string\n   score - Returns the score as a real number\n   IC - information content. Returns a real number\n   id - identifier. Returns a string\n   accession - accession number. Returns a string\n   next_pos - return the sequence probably for each letter, IUPAC\n         symbol, IUPAC probability and simple sequence\n   consenus letter for this position. Rewind at the end. Returns a hash.\n   pos - current position get/set. Returns an integer.\n   regexp - construct a regular expression based on IUPAC consensus.\n         For example AGWV will be [Aa][Gg][AaTt][AaCcGg]\n   width - site width\n   get_string - gets the probability vector for a single base as a string.\n   get_array - gets the probability vector for a single base as an array.\n   get_logs_array - gets the log-odds vector for a single base as an array.\n\nNew methods, which might be of interest to anyone who wants to store\nPSM in a relational database without creating an entry for each\nposition is the ability to compress the PSM vector into a string with\nlosing usually less than 1% of the data.  this can be done with:\n\n   my $str=$matrix->get_compressed_freq('A');\nor\n\n   my $str=$matrix->get_compressed_logs('A');\n\nLoading from a database should be done with new, but is not yet implemented.\nHowever you can still uncompress such string with:\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1,1); for PSM\n\nor\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1000,2); for log odds\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to one\nof the Bioperl mailing lists.   Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nthe bugs and their resolution.  Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 AUTHOR - James Thompson\n\nEmail tex@biosysadmin.com\n\n=head1 APPENDIX\n\n\n\n# Let the code begin...\npackage Bio::Matrix::PSM::ProtMatrix;\nuse strict;\n\nuse base qw(Bio::Root::Root Bio::Matrix::PSM::SiteMatrixI);\n\n=head2 new\n\n Title    : new\n Usage    : my $site = Bio::Matrix::PSM::ProtMatrix->new( \n               %probs,\n               %logs,\n               -IC    => $ic,\n               -e_val => $score, \n               -id    => $mid\n               -model => \\%model\n            );\n Function : Creates a new Bio::Matrix::PSM::ProtMatrix object from memory\n Throws   : If inconsistent data for all vectors (all 20 amino acids) is\n               provided, if you mix input types (string vs array) or if a\n               position freq is 0.\n Example  :\n Returns  : Bio::Matrix::PSM::ProtMatrix object\n Args     : Hash references to log-odds scores and probabilities for\n            position-specific scoring info, e-value (optional), information\n            content (optional), id (optional), model for background distribution\n            of proteins (optional)."},"containerName":"main::","definition":"sub","line":218,"children":[{"line":219,"kind":13,"localvar":"my","containerName":"new","definition":"my","name":"$class"},{"containerName":"new","name":"@args","line":219,"kind":13},{"line":220,"kind":13,"localvar":"my","containerName":"new","name":"$self","definition":"my"},{"name":"$class","containerName":"new","line":220,"kind":13},{"name":"@args","containerName":"new","line":220,"kind":13},{"line":221,"kind":13,"localvar":"my","containerName":"new","definition":"my","name":"$consensus"},{"kind":13,"line":224,"definition":"my","name":"%input","containerName":"new","localvar":"my"},{"kind":13,"line":225,"name":"@args","containerName":"new"},{"localvar":"my","name":"$key","definition":"my","containerName":"new","line":226,"kind":13},{"containerName":"new","name":"@args","kind":13,"line":226},{"line":227,"kind":13,"name":"$input","containerName":"new"},{"kind":13,"line":227,"name":"$key","containerName":"new"},{"containerName":"new","name":"@args","line":227,"kind":13},{"localvar":"my","containerName":"new","definition":"my","name":"@alphabet","line":232,"kind":13},{"definition":"my","name":"$aa","containerName":"new","localvar":"my","kind":13,"line":234},{"kind":13,"line":234,"name":"@alphabet","containerName":"new"},{"name":"$self","containerName":"new","kind":13,"line":235},{"line":235,"kind":13,"containerName":"new","name":"$input"},{"kind":13,"line":235,"containerName":"new","name":"$input"},{"kind":13,"line":236,"name":"$self","containerName":"new"},{"line":236,"kind":12,"containerName":"new","name":"throw"},{"name":"$self","containerName":"new","line":237,"kind":13},{"kind":13,"line":237,"containerName":"new","name":"$input"},{"kind":13,"line":237,"name":"$input","containerName":"new"},{"name":"$self","containerName":"new","kind":13,"line":238},{"containerName":"new","name":"throw","kind":12,"line":238},{"line":241,"kind":13,"containerName":"new","name":"$self"},{"line":242,"kind":13,"name":"$self","containerName":"new"},{"containerName":"new","name":"$input","kind":13,"line":242},{"kind":13,"line":243,"name":"$self","containerName":"new"},{"containerName":"new","name":"$input","line":243,"kind":13},{"line":244,"kind":13,"containerName":"new","name":"$self"},{"line":244,"kind":13,"containerName":"new","name":"$input"},{"containerName":"new","name":"$self","line":245,"kind":13},{"name":"$input","containerName":"new","kind":13,"line":245},{"kind":13,"line":246,"name":"$self","containerName":"new"},{"name":"$input","containerName":"new","line":246,"kind":13},{"line":247,"kind":13,"name":"$self","containerName":"new"},{"containerName":"new","name":"$input","kind":13,"line":247},{"kind":13,"line":248,"containerName":"new","name":"$input"},{"containerName":"new","name":"$self","kind":13,"line":250},{"containerName":"new","name":"$input","kind":13,"line":250},{"name":"$input","containerName":"new","line":250,"kind":13},{"containerName":"new","name":"$self","kind":13,"line":251},{"name":"@alphabet","containerName":"new","line":251,"kind":13},{"containerName":"new","name":"$self","line":254,"kind":13},{"line":254,"kind":13,"containerName":"new","name":"$self"},{"containerName":"new","name":"$input","kind":13,"line":254},{"line":255,"kind":13,"name":"$self","containerName":"new"}],"kind":12,"range":{"end":{"character":9999,"line":256},"start":{"line":218,"character":0}},"name":"new"},{"containerName":"new","name":"SUPER","line":220,"kind":12},{"name":"_position","kind":12,"line":241},{"kind":12,"line":242,"name":"IC"},{"line":242,"kind":12,"name":"IC"},{"name":"e_val","line":243,"kind":12},{"kind":12,"line":243,"name":"e_val"},{"kind":12,"line":244,"name":"sites"},{"name":"sites","line":244,"kind":12},{"kind":12,"line":245,"name":"width"},{"name":"width","line":245,"kind":12},{"kind":12,"line":246,"name":"accession_number"},{"kind":12,"line":246,"name":"accession_number"},{"name":"_correction","line":247,"kind":12},{"line":247,"kind":12,"name":"correction"},{"name":"correction","line":248,"kind":12},{"name":"id","kind":12,"line":250},{"name":"id","line":250,"kind":12},{"name":"id","kind":12,"line":250},{"name":"_alphabet","kind":12,"line":251},{"kind":12,"line":254,"name":"_calculate_consensus"},{"name":"model","line":254,"kind":12},{"children":[{"localvar":"my","containerName":"alphabet","definition":"my","name":"$self","line":272,"kind":13},{"kind":13,"line":274,"containerName":"alphabet","name":"$self"},{"containerName":"alphabet","name":"$self","line":276,"kind":13}],"line":271,"kind":12,"range":{"start":{"character":0,"line":271},"end":{"character":9999,"line":277}},"name":"alphabet","definition":"sub","containerName":"main::"},{"kind":12,"line":274,"name":"_alphabet"},{"line":276,"kind":12,"name":"_alphabet"},{"name":"_calculate_consensus","definition":"sub","containerName":"main::","range":{"end":{"character":9999,"line":297},"start":{"line":292,"character":0}},"kind":12,"children":[{"localvar":"my","containerName":"_calculate_consensus","name":"$self","definition":"my","line":293,"kind":13},{"definition":"my","name":"$thresh","containerName":"_calculate_consensus","localvar":"my","kind":13,"line":294},{"containerName":"_calculate_consensus","definition":"my","name":"@lengths","localvar":"my","kind":13,"line":297},{"kind":13,"line":297,"name":"$_","containerName":"_calculate_consensus"},{"kind":13,"line":297,"containerName":"_calculate_consensus","name":"$self"},{"name":"$self","containerName":"_calculate_consensus","kind":13,"line":297}],"line":292},{"line":297,"kind":12,"name":"_alphabet"},{"containerName":null,"definition":"my","name":"$len","localvar":"my","kind":13,"line":298},{"name":"@lengths","containerName":null,"line":298,"kind":13},{"line":299,"kind":13,"name":"@lengths","containerName":null},{"kind":13,"line":300,"name":"%len","containerName":null},{"line":300,"kind":13,"name":"$self","containerName":null},{"name":"throw","containerName":"main::","kind":12,"line":300},{"kind":13,"line":305,"definition":"my","name":"$i","containerName":null,"localvar":"my"},{"name":"$i","containerName":null,"kind":13,"line":305},{"containerName":null,"name":"$len","line":305,"kind":13},{"kind":13,"line":305,"name":"%i","containerName":null},{"line":307,"kind":13,"localvar":"my","containerName":null,"name":"@probs","definition":"my"},{"kind":13,"line":307,"name":"%self","containerName":null},{"name":"$i","containerName":null,"line":307,"kind":13},{"containerName":null,"name":"%self","kind":13,"line":307},{"name":"_alphabet","kind":12,"line":307},{"kind":13,"line":309,"containerName":null,"name":"%self"},{"line":309,"kind":12,"name":"seq"},{"name":"$i","containerName":null,"kind":13,"line":309},{"kind":13,"line":309,"name":"%self","containerName":null},{"line":309,"kind":12,"name":"seqp"},{"name":"$i","containerName":null,"kind":13,"line":309},{"containerName":null,"name":"$self","line":309,"kind":13},{"containerName":"main::","name":"_to_cons","kind":12,"line":309},{"containerName":null,"name":"@probs","kind":13,"line":309},{"line":309,"kind":13,"containerName":null,"name":"$thresh"},{"name":"$self","containerName":null,"line":312,"kind":13},{"kind":12,"line":340,"children":[{"localvar":"my","name":"$self","definition":"my","containerName":"next_pos","line":341,"kind":13},{"kind":13,"line":342,"name":"$self","containerName":"next_pos"},{"kind":12,"line":342,"containerName":"next_pos","name":"throw"},{"line":342,"kind":13,"containerName":"next_pos","name":"$self"},{"localvar":"my","containerName":"next_pos","definition":"my","name":"$len","line":344,"kind":13},{"name":"$self","containerName":"next_pos","kind":13,"line":344}],"containerName":"main::","name":"next_pos","definition":"sub","range":{"start":{"character":0,"line":340},"end":{"line":344,"character":9999}}},{"line":344,"kind":12,"name":"seq"},{"line":345,"kind":13,"localvar":"my","containerName":null,"name":"$pos","definition":"my"},{"kind":13,"line":345,"name":"%self","containerName":null},{"name":"_position","kind":12,"line":345},{"line":348,"kind":13,"name":"$pos","containerName":null},{"line":348,"kind":13,"containerName":null,"name":"%len"},{"localvar":"my","containerName":null,"definition":"my","name":"%probs","line":350,"kind":13},{"containerName":null,"name":"%self","line":350,"kind":13},{"containerName":null,"name":"$pos","line":350,"kind":13},{"containerName":null,"name":"%self","line":350,"kind":13},{"line":350,"kind":12,"name":"_alphabet"},{"containerName":null,"name":"%logs","definition":"my","localvar":"my","kind":13,"line":351},{"containerName":null,"name":"%self","kind":13,"line":351},{"line":351,"kind":13,"containerName":null,"name":"$pos"},{"name":"%self","containerName":null,"kind":13,"line":351},{"kind":12,"line":351,"name":"_alphabet"},{"localvar":"my","name":"$base","definition":"my","containerName":null,"line":352,"kind":13},{"line":352,"kind":13,"containerName":null,"name":"%self"},{"kind":12,"line":352,"name":"seq"},{"line":352,"kind":13,"containerName":null,"name":"$pos"},{"localvar":"my","name":"$prob","definition":"my","containerName":null,"line":353,"kind":13},{"line":353,"kind":13,"containerName":null,"name":"%self"},{"name":"seqp","kind":12,"line":353},{"containerName":null,"name":"$pos","kind":13,"line":353},{"containerName":null,"name":"%self","kind":13,"line":355},{"name":"_position","line":355,"kind":12},{"localvar":"my","name":"%hash","definition":"my","containerName":null,"line":356,"kind":13},{"name":"%probs","containerName":null,"kind":13,"line":356},{"kind":13,"line":356,"containerName":null,"name":"%logs"},{"name":"base","line":356,"kind":12},{"kind":13,"line":356,"containerName":null,"name":"$base"},{"line":356,"kind":12,"name":"rel"},{"containerName":null,"name":"$pos","kind":13,"line":356},{"kind":12,"line":356,"name":"prob"},{"kind":13,"line":356,"containerName":null,"name":"%prob"},{"name":"%hash","containerName":null,"line":360,"kind":13},{"kind":13,"line":362,"name":"%hash","containerName":null},{"line":365,"kind":13,"name":"%self","containerName":null},{"name":"_position","kind":12,"line":365},{"definition":"sub","name":"curpos","containerName":"main::","range":{"end":{"character":9999,"line":387},"start":{"character":0,"line":383}},"kind":12,"children":[{"localvar":"my","definition":"my","name":"$self","containerName":"curpos","line":384,"kind":13},{"line":385,"kind":13,"name":"$self","containerName":"curpos"},{"line":386,"kind":13,"name":"$self","containerName":"curpos"}],"line":383},{"kind":12,"line":385,"name":"_position"},{"name":"_position","line":386,"kind":12},{"children":[{"line":403,"kind":13,"localvar":"my","containerName":"e_val","name":"$self","definition":"my"},{"kind":13,"line":404,"name":"$self","containerName":"e_val"},{"name":"$self","containerName":"e_val","line":405,"kind":13}],"line":402,"kind":12,"range":{"end":{"line":406,"character":9999},"start":{"character":0,"line":402}},"name":"e_val","definition":"sub","containerName":"main::"},{"name":"e_val","line":404,"kind":12},{"kind":12,"line":405,"name":"e_val"},{"line":421,"children":[{"definition":"my","name":"$self","containerName":"IC","localvar":"my","kind":13,"line":422},{"kind":13,"line":423,"containerName":"IC","name":"$self"},{"kind":13,"line":424,"containerName":"IC","name":"$self"}],"kind":12,"range":{"start":{"line":421,"character":0},"end":{"character":9999,"line":425}},"containerName":"main::","name":"IC","definition":"sub"},{"name":"IC","kind":12,"line":423},{"line":424,"kind":12,"name":"IC"},{"range":{"start":{"line":440,"character":0},"end":{"character":9999,"line":444}},"containerName":"main::","name":"accession_number","definition":"sub","line":440,"children":[{"kind":13,"line":441,"definition":"my","name":"$self","containerName":"accession_number","localvar":"my"},{"line":442,"kind":13,"name":"$self","containerName":"accession_number"},{"kind":13,"line":443,"containerName":"accession_number","name":"$self"}],"kind":12},{"name":"accession_number","kind":12,"line":442},{"line":443,"kind":12,"name":"accession_number"},{"children":[{"name":"$self","definition":"my","containerName":"consensus","localvar":"my","kind":13,"line":459},{"containerName":"consensus","definition":"my","name":"$thresh","localvar":"my","kind":13,"line":460},{"containerName":"consensus","name":"$self","kind":13,"line":461},{"kind":12,"line":461,"name":"_calculate_consensus","containerName":"consensus"},{"containerName":"consensus","name":"$thresh","line":461,"kind":13},{"line":461,"kind":13,"containerName":"consensus","name":"$thresh"},{"localvar":"my","containerName":"consensus","name":"$consensus","definition":"my","line":462,"kind":13},{"line":464,"kind":13,"localvar":"my","containerName":"consensus","definition":"my","name":"$letter"},{"line":464,"kind":13,"containerName":"consensus","name":"$self"}],"line":458,"kind":12,"range":{"start":{"character":0,"line":458},"end":{"character":9999,"line":464}},"definition":"sub","name":"consensus","containerName":"main::"},{"name":"seq","line":464,"kind":12},{"name":"$consensus","containerName":null,"kind":13,"line":465},{"line":465,"kind":13,"containerName":null,"name":"$letter"},{"kind":13,"line":468,"containerName":null,"name":"$consensus"},{"range":{"end":{"character":9999,"line":474},"start":{"character":0,"line":471}},"containerName":"main::","name":"IUPAC","definition":"sub","line":471,"children":[{"localvar":"my","containerName":"IUPAC","name":"$self","definition":"my","line":472,"kind":13},{"containerName":"IUPAC","name":"$self","kind":13,"line":473},{"containerName":"IUPAC","name":"consensus","kind":12,"line":473}],"kind":12},{"containerName":"main::","name":"get_string","definition":"sub","range":{"start":{"character":0,"line":490},"end":{"line":495,"character":9999}},"kind":12,"line":490,"children":[{"localvar":"my","containerName":"get_string","name":"$self","definition":"my","line":491,"kind":13},{"name":"$base","definition":"my","containerName":"get_string","localvar":"my","kind":13,"line":492},{"localvar":"my","containerName":"get_string","name":"$string","definition":"my","line":493,"kind":13},{"line":495,"kind":13,"localvar":"my","containerName":"get_string","name":"@prob","definition":"my"},{"containerName":"get_string","name":"$self","kind":13,"line":495}]},{"containerName":null,"name":"@prob","kind":13,"line":496},{"containerName":null,"name":"$self","line":497,"kind":13},{"kind":12,"line":497,"name":"throw","containerName":"main::"},{"line":500,"kind":13,"localvar":"my","definition":"my","name":"$prob","containerName":null},{"name":"@prob","containerName":null,"kind":13,"line":500},{"localvar":"my","containerName":null,"definition":"my","name":"$corrected","line":501,"kind":13},{"line":501,"kind":13,"name":"$prob","containerName":null},{"localvar":"my","name":"$next","definition":"my","containerName":null,"line":502,"kind":13},{"name":"$corrected","containerName":null,"line":502,"kind":13},{"kind":13,"line":503,"name":"$next","containerName":null},{"line":503,"kind":13,"name":"$next","containerName":null},{"containerName":null,"name":"$string","line":504,"kind":13},{"line":504,"kind":13,"containerName":null,"name":"$next"},{"line":506,"kind":13,"containerName":null,"name":"$string"},{"range":{"end":{"character":9999,"line":525},"start":{"character":0,"line":523}},"name":"width","definition":"sub","containerName":"main::","children":[{"localvar":"my","containerName":"width","name":"$self","definition":"my","line":524,"kind":13},{"localvar":"my","name":"$width","definition":"my","containerName":"width","line":525,"kind":13},{"containerName":"width","name":"$self","kind":13,"line":525}],"line":523,"kind":12},{"line":525,"kind":12,"name":"probA"},{"kind":13,"line":526,"containerName":null,"name":"$width"},{"containerName":"main::","name":"get_array","definition":"sub","range":{"start":{"line":541,"character":0},"end":{"line":545,"character":9999}},"kind":12,"line":541,"children":[{"line":542,"kind":13,"localvar":"my","containerName":"get_array","definition":"my","name":"$self"},{"containerName":"get_array","name":"$letter","definition":"my","localvar":"my","kind":13,"line":543},{"containerName":"get_array","name":"$self","kind":13,"line":545},{"name":"throw","containerName":"get_array","kind":12,"line":545},{"containerName":"get_array","name":"$self","line":545,"kind":13}]},{"name":"_alphabet","line":545,"kind":12},{"containerName":null,"name":"%self","line":547,"kind":13},{"kind":12,"line":563,"children":[{"name":"$self","definition":"my","containerName":"get_logs_array","localvar":"my","kind":13,"line":564},{"line":565,"kind":13,"localvar":"my","definition":"my","name":"$letter","containerName":"get_logs_array"},{"kind":13,"line":567,"containerName":"get_logs_array","name":"$self"},{"name":"throw","containerName":"get_logs_array","line":567,"kind":12},{"containerName":"get_logs_array","name":"$self","kind":13,"line":567}],"containerName":"main::","name":"get_logs_array","definition":"sub","range":{"end":{"character":9999,"line":567},"start":{"character":0,"line":563}}},{"name":"_alphabet","line":567,"kind":12},{"line":569,"kind":13,"containerName":null,"name":"%self"},{"kind":12,"children":[{"containerName":"id","name":"$self","definition":"my","localvar":"my","kind":13,"line":585},{"name":"$self","containerName":"id","kind":13,"line":586},{"kind":13,"line":587,"containerName":"id","name":"$self"}],"line":584,"definition":"sub","name":"id","containerName":"main::","range":{"start":{"line":584,"character":0},"end":{"character":9999,"line":588}}},{"line":586,"kind":12,"name":"id"},{"name":"id","kind":12,"line":587},{"definition":"sub","name":"regexp","containerName":"main::","range":{"end":{"line":609,"character":9999},"start":{"character":0,"line":604}},"kind":12,"children":[{"name":"$self","definition":"my","containerName":"regexp","localvar":"my","kind":13,"line":605},{"containerName":"regexp","definition":"my","name":"$threshold","localvar":"my","kind":13,"line":606},{"containerName":"regexp","name":"$threshold","definition":"my","localvar":"my","kind":13,"line":607},{"line":609,"kind":13,"localvar":"my","containerName":"regexp","name":"@alphabet","definition":"my"},{"line":609,"kind":13,"name":"$self","containerName":"regexp"}],"line":604},{"name":"_alphabet","line":609,"kind":12},{"kind":13,"line":610,"containerName":null,"definition":"my","name":"$width","localvar":"my"},{"containerName":null,"name":"$self","line":610,"kind":13},{"kind":12,"line":610,"name":"width","containerName":"main::"},{"line":611,"kind":13,"localvar":"my","containerName":null,"definition":"my","name":"@regexp"},{"name":"$i","containerName":null,"line":611,"kind":13},{"line":612,"kind":13,"containerName":null,"name":"$i"},{"kind":13,"line":612,"name":"$i","containerName":null},{"name":"$width","containerName":null,"line":612,"kind":13},{"line":612,"kind":13,"containerName":null,"name":"%i"},{"line":614,"kind":13,"localvar":"my","containerName":null,"definition":"my","name":"@letters"},{"line":614,"kind":13,"containerName":null,"name":"%self"},{"kind":13,"line":614,"name":"$i","containerName":null},{"name":"$threshold","containerName":null,"line":614,"kind":13},{"line":614,"kind":13,"containerName":null,"name":"@alphabet"},{"line":616,"kind":13,"localvar":"my","name":"$reg","definition":"my","containerName":null},{"name":"@letters","containerName":null,"kind":13,"line":617},{"name":"%reg","containerName":null,"kind":13,"line":618},{"containerName":null,"name":"$reg","kind":13,"line":620},{"line":620,"kind":13,"containerName":null,"name":"@letters"},{"kind":13,"line":622,"name":"@regexp","containerName":null},{"name":"%reg","containerName":null,"line":622,"kind":13},{"name":"@regexp","containerName":null,"line":626,"kind":13},{"containerName":null,"name":"@regexp","kind":13,"line":628},{"definition":"sub","name":"regexp_array","containerName":"main::","range":{"start":{"character":0,"line":648},"end":{"line":651,"character":9999}},"kind":12,"children":[{"definition":"my","name":"$self","containerName":"regexp_array","localvar":"my","kind":13,"line":649},{"kind":13,"line":651,"containerName":"regexp_array","name":"$self"},{"name":"regexp","containerName":"regexp_array","line":651,"kind":12}],"line":648},{"range":{"start":{"character":0,"line":669},"end":{"line":675,"character":9999}},"name":"_compress_array","signature":{"label":"_compress_array($array,$lm,$direct)","parameters":[{"label":"$array"},{"label":"$lm"},{"label":"$direct"}],"documentation":"1;\n# $Id: ProtMatrix.pm 16123 2009-09-17 12:57:27Z cjfields $\n#---------------------------------------------------------\n\n=head1 NAME\n\nBio::Matrix::PSM::ProtMatrix - SiteMatrixI implementation, holds a\nposition scoring matrix (or position weight matrix) with log-odds scoring\ninformation.\n\n=head1 SYNOPSIS\n\n   use Bio::Matrix::PSM::ProtMatrix;\n   # Create from memory by supplying probability matrix hash both as strings or\n   # arrays where the frequencies   Hash entries of the form lN refer to an array\n   # of position-specific log-odds scores for amino acid N. Hash entries of the\n   # form pN represent the position-specific probability of finding amino acid N.\n\n   my %param = (\n             'id' => 'A. thaliana protein atp1',\n             '-e_val' => $score,\n             'lS' => [ '-2', '3', '-3', '2', '-3', '1', '1', '3' ],\n             'lF' => [ '-1', '-4', '0', '-5', '0', '-5', '-4', '-4' ],\n             'lT' => [ '-1', '1', '0', '1', '-2', '-1', '0', '1' ],\n             'lN' => [ '-3', '-1', '-2', '3', '-5', '5', '-2', '0' ],\n             'lK' => [ '-2', '0', '-3', '2', '-3', '2', '-3', '-1' ],\n             'lY' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-4', '-4' ],\n             'lE' => [ '-3', '4', '-3', '2', '-4', '-2', '-3', '2' ],\n             'lV' => [ '0', '-2', '1', '-4', '1', '-4', '-1', '-3' ],\n             'lQ' => [ '-1', '0', '-2', '3', '-4', '1', '-3', '0' ],\n             'lM' => [ '8', '-3', '8', '-3', '1', '-3', '-3', '-3' ],\n             'lC' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-3', '-3' ],\n             'lL' => [ '1', '-3', '1', '-4', '3', '-4', '-2', '-4' ],\n             'lA' => [ '-2', '1', '-2', '0', '-2', '-2', '2', '2' ],\n             'lW' => [ '-2', '-4', '-3', '-5', '-4', '-5', '-5', '-5' ],\n             'lP' => [ '-3', '-2', '-4', '-3', '-1', '-3', '6', '-3' ],\n             'lH' => [ '-2', '-2', '-3', '-2', '-5', '-2', '-2', '-3' ],\n             'lD' => [ '-4', '-1', '-3', '1', '-3', '-1', '-3', '4' ],\n             'lR' => [ '-2', '-1', '-3', '0', '-4', '4', '-4', '-3' ],\n             'lI' => [ '0', '-3', '0', '-4', '6', '-4', '-2', '-2' ],\n             'lG' => [ '-4', '-2', '-4', '-2', '-5', '-3', '-1', '-2' ],\n             'pS' => [ '0', '33', '0', '16', '1', '12', '11', '25' ],\n             'pF' => [ '0', '0', '2', '0', '3', '0', '0', '0' ],\n             'pT' => [ '0', '8', '7', '10', '1', '2', '7', '8' ],\n             'pN' => [ '0', '0', '2', '13', '0', '36', '1', '4' ],\n             'pK' => [ '0', '5', '0', '13', '1', '15', '0', '2' ],\n             'pY' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pE' => [ '0', '41', '1', '12', '0', '0', '0', '15' ],\n             'pV' => [ '0', '3', '9', '0', '2', '0', '3', '1' ],\n             'pQ' => [ '0', '0', '0', '15', '0', '4', '0', '3' ],\n             'pM' => [ '100', '0', '66', '0', '2', '0', '0', '0' ],\n             'pC' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pL' => [ '0', '0', '8', '0', '25', '0', '4', '0' ],\n             'pA' => [ '0', '10', '1', '9', '2', '0', '22', '16' ],\n             'pW' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pP' => [ '0', '0', '0', '0', '3', '1', '45', '0' ],\n             'pH' => [ '0', '0', '0', '0', '0', '0', '1', '0' ],\n             'pD' => [ '0', '0', '1', '7', '2', '2', '0', '22' ],\n             'pR' => [ '0', '0', '0', '3', '0', '27', '0', '0' ],\n             'pI' => [ '0', '0', '3', '0', '59', '1', '2', '3' ],\n             'pG' => [ '0', '0', '0', '1', '0', '0', '4', '1' ],\n   );\n\n   my $matrix = Bio::Matrix::PSM::ProtMatrix( %param );\n\n\n   my $site = Bio::Matrix::PSM::ProtMatrix->new(%param);\n   # Or get it from a file:\n   use Bio::Matrix::PSM::IO;\n   my $psmIO = Bio::Matrix::PSM::IO->new(-file => $file, -format => 'psi-blast');\n   while (my $psm = $psmIO->next_psm) {\n      #Now we have a Bio::Matrix::PSM::Psm object, \n      # see Bio::Matrix::PSM::PsmI for details\n      #This is a Bio::Matrix::PSM::ProtMatrix object now\n      my $matrix = $psm->matrix;   \n   }\n\n   # Get a simple consensus, where alphabet is:\n   # {A, R, N, D, C, Q, E, G, H, I, L, K, M, F, P, S, T, W, Y, V,}\n   # choosing the highest probability or N if prob is too low\n   my $consensus = $site->consensus;\n\n   # Retrieving and using regular expressions:\n   my $regexp = $site->regexp;\n   my $count = grep($regexp,$seq);\n   my $count = ($seq=~ s/$regexp/$1/eg);\n   print \"Motif $mid is present $count times in this sequence\\n\";\n\n=head1 DESCRIPTION\n\nProtMatrix is designed to provide some basic methods when working with\nposition scoring (weight) matrices related to protein sequences.  A\nprotein PSM consists of 20 vectors with 20 frequencies (one per amino\nacid per position).  This is the minimum information you should\nprovide to construct a PSM object.  The vectors can be provided as\nstrings with frequencies where the frequency is {0..a} and a=1. This\nis the way MEME compressed representation of a matrix and it is quite\nuseful when working with relational DB.  If arrays are provided as an\ninput (references to arrays actually) they can be any number, real or\ninteger (frequency or count).\n\nWhen creating the object the constructor will check for positions that\nequal 0.  If such is found it will increase the count for all\npositions by one and recalculate the frequency.  Potential bug - if\nyou are using frequencies and one of the positions is 0 it will change\nsignificantly.  However, you should never have frequency that equals\n0.\n\nThrows an exception if: You mix as an input array and string (for\nexample A matrix is given as array, C - as string).  The position\nvector is (0,0,0,0).  One of the probability vectors is shorter than\nthe rest.\n\nSummary of the methods I use most frequently (details bellow):\n\n   iupac - return IUPAC compliant consensus as a string\n   score - Returns the score as a real number\n   IC - information content. Returns a real number\n   id - identifier. Returns a string\n   accession - accession number. Returns a string\n   next_pos - return the sequence probably for each letter, IUPAC\n         symbol, IUPAC probability and simple sequence\n   consenus letter for this position. Rewind at the end. Returns a hash.\n   pos - current position get/set. Returns an integer.\n   regexp - construct a regular expression based on IUPAC consensus.\n         For example AGWV will be [Aa][Gg][AaTt][AaCcGg]\n   width - site width\n   get_string - gets the probability vector for a single base as a string.\n   get_array - gets the probability vector for a single base as an array.\n   get_logs_array - gets the log-odds vector for a single base as an array.\n\nNew methods, which might be of interest to anyone who wants to store\nPSM in a relational database without creating an entry for each\nposition is the ability to compress the PSM vector into a string with\nlosing usually less than 1% of the data.  this can be done with:\n\n   my $str=$matrix->get_compressed_freq('A');\nor\n\n   my $str=$matrix->get_compressed_logs('A');\n\nLoading from a database should be done with new, but is not yet implemented.\nHowever you can still uncompress such string with:\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1,1); for PSM\n\nor\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1000,2); for log odds\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to one\nof the Bioperl mailing lists.   Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nthe bugs and their resolution.  Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 AUTHOR - James Thompson\n\nEmail tex@biosysadmin.com\n\n=head1 APPENDIX\n\n\n\n# Let the code begin...\npackage Bio::Matrix::PSM::ProtMatrix;\nuse strict;\n\nuse base qw(Bio::Root::Root Bio::Matrix::PSM::SiteMatrixI);\n\n=head2 new\n\n Title    : new\n Usage    : my $site = Bio::Matrix::PSM::ProtMatrix->new( \n               %probs,\n               %logs,\n               -IC    => $ic,\n               -e_val => $score, \n               -id    => $mid\n               -model => \\%model\n            );\n Function : Creates a new Bio::Matrix::PSM::ProtMatrix object from memory\n Throws   : If inconsistent data for all vectors (all 20 amino acids) is\n               provided, if you mix input types (string vs array) or if a\n               position freq is 0.\n Example  :\n Returns  : Bio::Matrix::PSM::ProtMatrix object\n Args     : Hash references to log-odds scores and probabilities for\n            position-specific scoring info, e-value (optional), information\n            content (optional), id (optional), model for background distribution\n            of proteins (optional).\n\n\nsub new {\n   my ($class, @args) = @_;\n   my $self = $class->SUPER::new(@args);\n   my $consensus;\n   #Too many things to rearrange, and I am creating simultanuously >500 \n   # such objects routinely, so this becomes performance issue\n   my %input;\n   while( @args ) {\n     (my $key = shift @args) =~ s/-//gi; #deletes all dashes (only dashes)!\n     $input{$key} = shift @args;\n   }\n\n   # get a protein alphabet for processing log-odds scores and probabilities\n   # maybe change this later on to allow for non-standard aa lists? \n   my @alphabet = qw/A  R  N  D  C  Q  E  G  H  I  L  K  M  F  P  S  T  W  Y  V/;\n\n   foreach my $aa (@alphabet) {\n      $self->{\"log$aa\"}  = defined($input{\"l$aa\"}) ? $input{\"l$aa\"} \n         : $self->throw(\"Error: No log-odds information for $aa!\");\n      $self->{\"prob$aa\"}  = defined($input{\"p$aa\"}) ? $input{\"p$aa\"} \n         : $self->throw(\"Error: No probability information for $aa!\");\n   }\n   \n   $self->{_position} = 0;\n   $self->{IC}        = $input{IC};\n   $self->{e_val}     = $input{e_val};\n   $self->{sites}     = $input{sites};\n   $self->{width}     = $input{width};\n   $self->{accession_number} = $input{accession_number};\n   $self->{_correction} = defined($input{correction}) ?\n      $input{correction} : 1 ; # Correction might be unwanted- supply your own\n   # No id provided, null for the sake of rel db\n   $self->{id} = defined($input{id}) ? $input{id} : 'null'; \n   $self->{_alphabet} = \\@alphabet;\n\n   #Make consensus, throw if any one of the vectors is shorter\n   $self = _calculate_consensus($self,$input{model});\n   return $self;\n}\n\n=head2 alphabet\n\n Title    : Returns an array (or array reference if desired) to the alphabet \n Usage    :\n Function : Returns an array (or array reference) containing all of the\n            allowable characters for this matrix.\n Throws   :\n Example  :\n Returns  : Array or arrary reference.\n Args     :\n\n\nsub alphabet {\n   my $self = shift;\n   if ( wantarray ) {\n      return $self->{_alphabet};\n   } else {\n      return @{$self->{_alphabet}};\n   }\n}\n\n=head2 _calculate_consensus\n\n Title    : _calculate_consensus\n Usage    :\n Function : Calculates the consensus sequence for this matrix. \n Throws   :\n Example  :\n Returns  :\n Args     :\n\n\nsub _calculate_consensus {\n   my $self   = shift;\n   my $thresh = shift;\n   \n   # verify that all of the array lengths in %probs are the same\n   my @lengths = map { scalar(@$_) } map {$self->{\"prob$_\"}} @{ $self->{_alphabet} };\n   my $len = shift @lengths;\n   for ( @lengths ) {\n      if ( $_ ne $len ) { $self->throw( \"Probability matrix is damaged!\\n\" ) };\n   }\n\n   # iterate over probs, generate the most likely sequence and put it into \n   # $self->{seq}. Put the probability of this sequence into $self->{seqp}.\n   for ( my $i = 0; $i < $len; $i++ ) {\n      # get a list of all the probabilities at position $i, ordered by $self->{_alphabet}\n      my @probs = map { ${$self->{\"prob$_\"}}[$i] } @{ $self->{_alphabet} };\n      # calculate the consensus of @probs, put sequence into seqp and probabilities into seqp\n      (${$self->{seq}}[$i],${$self->{seqp}}[$i]) = $self->_to_cons( @probs, $thresh );\n   }\n\n   return $self;\n}\n\n=head2 next_pos\n\n Title    : next_pos\n Usage    :\n Function : Retrives the next position features: frequencies for all 20 amino\n            acids, log-odds scores for all 20 amino acids at this position,\n            the main (consensus) letter at this position, the probability\n            for the consensus letter to occur at this position and the relative\n            current position as an integer.\n Throws   :\n Example  :\n Returns  : hash (or hash reference) (pA,pR,pN,pD,...,logA,logR,logN,logD,aa,prob,rel)\n            - pN entries represent the probability for amino acid N\n            to be at this position\n            - logN entries represent the log-odds score for having amino acid\n            N at this position\n            - aa is the consensus amino acid\n            - prob is the probability for the consensus amino acid to be at this \n            position\n            - rel is the relative index of the current position (integer)\n Args      : none\n\n\n\nsub next_pos {\n      my $self = shift;\n         $self->throw(\"instance method called on class\") unless ref $self;\n\n      my $len = @{$self->{seq}};\n      my $pos = $self->{_position};\n\n      # return a PSM if we're still within range\n      if ($pos<$len) {\n\n         my %probs = map { (\"p$_\", ${$self->{\"prob$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my %logs  = map { (\"l$_\", ${$self->{\"log$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my $base  = ${$self->{seq}}[$pos];\n         my $prob  = ${$self->{seqp}}[$pos];\n\n         $self->{_position}++;\n         my %hash = ( %probs, %logs, base => $base, rel => $pos, prob => $prob );\n         \n         # decide whether to return the hash or a reference to it\n         if ( wantarray ) {\n            return %hash;\n         } else {\n            return \\%hash;\n         }\n   } else { # otherwise, reset $self->{_position} and return nothing\n      $self->{_position} = 0; \n      return;\n   }\n}\n\n\n=head2 curpos\n\n Title    : curpos\n Usage    :\n Function : Gets/sets the current position. \n Throws   :\n Example  :\n Returns  : Current position (integer).\n Args     : New position (integer).\n\n\nsub curpos {\n      my $self = shift;\n      if (@_) { $self->{_position} = shift; }\n      return $self->{_position};\n}\n\n\n=head2 e_val\n\n Title    : e_val\n Usage    :\n Function : Gets/sets the e-value\n Throws   :\n Example  :\n Returns  : \n Args     : real number\n\n\nsub e_val {\n      my $self = shift;\n      if (@_) { $self->{e_val} = shift; }\n      return $self->{e_val};\n}\n\n\n=head2 IC\n\n Title    : IC\n Usage    :\n Function : Position-specific information content.\n Throws   :\n Example  :\n Returns  : Information content for current position.\n Args     : Information content for current position.\n\n\nsub IC {\n      my $self = shift;\n      if (@_) { $self->{IC} = shift; }\n      return $self->{IC};\n}\n\n=head2 accession_number\n\n Title    : accession_number\n Usage    :\n Function: accession number, this will be unique id for the ProtMatrix object as\n            well for any other object, inheriting from ProtMatrix.\n Throws   :\n Example  :\n Returns  : New accession number (string)\n Args     : Accession number (string)\n\n\nsub accession_number {\n      my $self = shift;\n      if (@_) { $self->{accession_number} = shift; }\n      return $self->{accession_number};\n}\n\n=head2 consensus\n\n Title    : consensus\n Usage    :\n Function : Returns the consensus sequence for this PSM.\n Throws   : if supplied with thresold outisde 5..10 range\n Example  :\n Returns  : string\n Args     : (optional) threshold value 5 to 10 (corresponds to 50-100% at each position\n\n\nsub consensus {\n   my $self = shift;\n   my $thresh=shift;\n   $self->_calculate_consensus($thresh) if ($thresh); #Change of threshold\n   my $consensus='';\n\n   foreach my $letter (@{$self->{seq}}) {\n       $consensus .= $letter;\n   }\n\n   return $consensus;\n}\n\nsub IUPAC {\n   my $self = shift;\n   return $self->consensus;\n}\n\n\n=head2 get_string\n\n Title   : get_string\n Usage   :\n Function: Returns given probability vector as a string. Useful if you want to\n            store things in a rel database, where arrays are not first choice\n Throws  : If the argument is outside {A,C,G,T}\n Example :\n Returns : string\n Args    : character {A,C,G,T}\n\n\nsub get_string {\n   my $self = shift;\n   my $base = shift;\n   my $string = '';\n\n   my @prob = @{$self->{\"prob$base\"}};\n   if ( ! @prob ) {\n      $self->throw( \"No such base: $base\\n\");\n   }\n\n   foreach my $prob (@prob) {\n      my $corrected = $prob*10;\n      my $next = sprintf(\"%.0f\",$corrected);\n      $next = 'a' if ($next eq '10');\n      $string .= $next;\n   }\n   return $string;\n}\n\n\n\n=head2 width\n\n Title    : width\n Usage    :\n Function : Returns the length of the site\n Throws   :\n Example  :\n Returns  : number\n Args     :\n\n\nsub width {\n   my $self = shift;\n   my $width = @{$self->{probA}};\n   return $width;\n}\n\n=head2 get_array\n\n Title    : get_array\n Usage    :\n Function : Returns an array with frequencies for a specified amino acid.\n Throws   :\n Example  :\n Returns  : Array representing frequencies for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"prob$letter\"}}; \n}\n\n\n=head2 get_logs_array\n\n Title    : get_logs_array\n Usage    :\n Function : Returns an array with log_odds for a specified base\n Throws   :\n Example  :\n Returns  : Array representing log-odds scores for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_logs_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"log$letter\"}}; \n}\n\n=head2 id\n\n Title    : id\n Usage    :\n Function : Gets/sets the site id\n Throws   :\n Example  :\n Returns  : string\n Args     : string\n\n\nsub id {\n      my $self = shift;\n      if (@_) { $self->{id} = shift; }\n      return $self->{id};\n}\n\n=head2 regexp\n\n Title    : regexp\n Usage    :\n Function : Returns a case-insensitive regular expression which matches the\n            IUPAC convention.  X's in consensus sequence will match anything.     \n Throws   :\n Example  :\n Returns  : string\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n\n\nsub regexp {\n   my $self = shift;\n   my $threshold = 20;\n   if ( @_ ) { my $threshold = shift };\n\n   my @alphabet = @{$self->{_alphabet}};\n   my $width = $self->width;\n   my (@regexp, $i);\n   for ( $i = 0; $i < $width; $i++ ) {\n      # get an array of the residues at this position with p > $threshold\n      my @letters = map { uc($_).lc($_) } grep { $self->{\"prob$_\"}->[$i] >= $threshold } @alphabet;\n\n      my $reg;\n      if ( scalar(@letters) == 0 ) {\n         $reg = '\\.';\n      } else {\n         $reg = '['.join('',@letters).']';\n      }\n      push @regexp, $reg;\n   }\n\n   if ( wantarray ) { \n      return @regexp;\n   } else {\n      return join '', @regexp;\n   }\n}\n\n\n=head2 regexp_array\n\n Title    : regexp_array\n Usage    :\n Function : Returns an array of position-specific regular expressions.\n             X's in consensus sequence will match anything.      \n Throws   :\n Example  :\n Returns  : Array of position-specific regular expressions.\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n Notes    : Simply calls regexp method in list context.\n\n\nsub regexp_array {\n   my $self = shift;\n   \n   return @{ $self->regexp };\n}\n\n\n=head2 _compress_array\n\n Title    : _compress_array\n Usage    :\n Function :  Will compress an array of real signed numbers to a string (ie vector of bytes)\n             -127 to +127 for bi-directional(signed) and 0..255 for unsigned ;\n Throws   :\n Example  :  Internal stuff\n Returns  :  String\n Args     :  array reference, followed by max value and direction (optional, defaults to 1),\n             direction of 1 is unsigned, anything else is signed. "},"detail":"($array,$lm,$direct)","definition":"sub","containerName":"main::","children":[{"kind":13,"line":670,"name":"$array","definition":"my","containerName":"_compress_array","localvar":"my"},{"name":"$lm","containerName":"_compress_array","kind":13,"line":670},{"line":670,"kind":13,"containerName":"_compress_array","name":"$direct"},{"containerName":"_compress_array","definition":"my","name":"$str","localvar":"my","kind":13,"line":671},{"line":672,"kind":13,"name":"$array","containerName":"_compress_array"},{"containerName":"_compress_array","name":"$lm","kind":13,"line":672},{"kind":13,"line":673,"name":"$direct","containerName":"_compress_array"},{"line":673,"kind":13,"containerName":"_compress_array","name":"$direct"},{"kind":13,"line":674,"name":"$k1","definition":"my","containerName":"_compress_array","localvar":"my"},{"kind":13,"line":674,"containerName":"_compress_array","name":"$direct"},{"kind":13,"line":674,"name":"$lm","containerName":"_compress_array"},{"kind":13,"line":674,"containerName":"_compress_array","name":"$lm"},{"containerName":"_compress_array","definition":"my","name":"$c","localvar":"my","kind":13,"line":675},{"kind":13,"line":675,"name":"$array","containerName":"_compress_array"}],"line":669,"kind":12},{"kind":13,"line":676,"name":"$c","containerName":null},{"name":"$lm","containerName":null,"line":676,"kind":13},{"kind":13,"line":676,"containerName":null,"name":"$c"},{"name":"$lm","containerName":null,"kind":13,"line":676},{"kind":13,"line":677,"name":"$c","containerName":null},{"containerName":null,"name":"$lm","kind":13,"line":677},{"line":677,"kind":13,"name":"$c","containerName":null},{"kind":13,"line":677,"containerName":null,"name":"$lm"},{"line":677,"kind":13,"containerName":null,"name":"$direct"},{"line":678,"kind":13,"containerName":null,"name":"$c"},{"kind":13,"line":678,"containerName":null,"name":"$c"},{"kind":13,"line":678,"name":"$direct","containerName":null},{"localvar":"my","name":"$byte","definition":"my","containerName":null,"line":679,"kind":13},{"kind":13,"line":679,"name":"$k1","containerName":null},{"containerName":null,"name":"$c","line":679,"kind":13},{"containerName":null,"name":"$byte","kind":13,"line":680},{"containerName":null,"name":"$byte","kind":13,"line":680},{"kind":13,"line":680,"containerName":null,"name":"$direct"},{"kind":13,"line":681,"containerName":null,"definition":"my","name":"$char","localvar":"my"},{"line":681,"kind":13,"containerName":null,"name":"$byte"},{"line":682,"kind":13,"name":"$str","containerName":null},{"kind":13,"line":682,"containerName":null,"name":"$char"},{"name":"$str","containerName":null,"kind":13,"line":684},{"name":"_uncompress_string","range":{"start":{"line":701,"character":0},"end":{"line":715,"character":9999}},"kind":12,"children":[{"localvar":"my","definition":"my","name":"$str","containerName":"_uncompress_string","line":702,"kind":13},{"kind":13,"line":702,"containerName":"_uncompress_string","name":"$lm"},{"name":"$direct","containerName":"_uncompress_string","line":702,"kind":13},{"localvar":"my","containerName":"_uncompress_string","name":"@array","definition":"my","line":703,"kind":13},{"line":704,"kind":13,"containerName":"_uncompress_string","name":"$str"},{"line":704,"kind":13,"containerName":"_uncompress_string","name":"$lm"},{"kind":13,"line":705,"containerName":"_uncompress_string","name":"$direct"},{"kind":13,"line":705,"containerName":"_uncompress_string","name":"$direct"},{"line":706,"kind":13,"localvar":"my","containerName":"_uncompress_string","definition":"my","name":"$k1"},{"kind":13,"line":706,"containerName":"_uncompress_string","name":"$direct"},{"kind":13,"line":706,"containerName":"_uncompress_string","name":"$lm"},{"containerName":"_uncompress_string","name":"$lm","line":706,"kind":13},{"line":707,"kind":13,"localvar":"my","definition":"my","name":"$c","containerName":"_uncompress_string"},{"containerName":"_uncompress_string","name":"$str","line":707,"kind":13},{"containerName":"_uncompress_string","definition":"my","name":"$byte","localvar":"my","kind":13,"line":708},{"name":"$c","containerName":"_uncompress_string","kind":13,"line":708},{"kind":13,"line":709,"name":"$byte","containerName":"_uncompress_string"},{"line":709,"kind":13,"name":"$byte","containerName":"_uncompress_string"},{"line":709,"kind":13,"containerName":"_uncompress_string","name":"$direct"},{"containerName":"_uncompress_string","name":"$num","definition":"my","localvar":"my","kind":13,"line":710},{"containerName":"_uncompress_string","name":"$byte","line":710,"kind":13},{"containerName":"_uncompress_string","name":"$k1","line":710,"kind":13},{"containerName":"_uncompress_string","name":"@array","line":711,"kind":13},{"containerName":"_uncompress_string","name":"$num","line":711,"kind":13},{"line":714,"kind":13,"containerName":"_uncompress_string","name":"@array"}],"line":701,"definition":"sub","containerName":"main::","signature":{"parameters":[{"label":"$str"},{"label":"$lm"},{"label":"$direct"}],"documentation":"1;\n# $Id: ProtMatrix.pm 16123 2009-09-17 12:57:27Z cjfields $\n#---------------------------------------------------------\n\n=head1 NAME\n\nBio::Matrix::PSM::ProtMatrix - SiteMatrixI implementation, holds a\nposition scoring matrix (or position weight matrix) with log-odds scoring\ninformation.\n\n=head1 SYNOPSIS\n\n   use Bio::Matrix::PSM::ProtMatrix;\n   # Create from memory by supplying probability matrix hash both as strings or\n   # arrays where the frequencies   Hash entries of the form lN refer to an array\n   # of position-specific log-odds scores for amino acid N. Hash entries of the\n   # form pN represent the position-specific probability of finding amino acid N.\n\n   my %param = (\n             'id' => 'A. thaliana protein atp1',\n             '-e_val' => $score,\n             'lS' => [ '-2', '3', '-3', '2', '-3', '1', '1', '3' ],\n             'lF' => [ '-1', '-4', '0', '-5', '0', '-5', '-4', '-4' ],\n             'lT' => [ '-1', '1', '0', '1', '-2', '-1', '0', '1' ],\n             'lN' => [ '-3', '-1', '-2', '3', '-5', '5', '-2', '0' ],\n             'lK' => [ '-2', '0', '-3', '2', '-3', '2', '-3', '-1' ],\n             'lY' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-4', '-4' ],\n             'lE' => [ '-3', '4', '-3', '2', '-4', '-2', '-3', '2' ],\n             'lV' => [ '0', '-2', '1', '-4', '1', '-4', '-1', '-3' ],\n             'lQ' => [ '-1', '0', '-2', '3', '-4', '1', '-3', '0' ],\n             'lM' => [ '8', '-3', '8', '-3', '1', '-3', '-3', '-3' ],\n             'lC' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-3', '-3' ],\n             'lL' => [ '1', '-3', '1', '-4', '3', '-4', '-2', '-4' ],\n             'lA' => [ '-2', '1', '-2', '0', '-2', '-2', '2', '2' ],\n             'lW' => [ '-2', '-4', '-3', '-5', '-4', '-5', '-5', '-5' ],\n             'lP' => [ '-3', '-2', '-4', '-3', '-1', '-3', '6', '-3' ],\n             'lH' => [ '-2', '-2', '-3', '-2', '-5', '-2', '-2', '-3' ],\n             'lD' => [ '-4', '-1', '-3', '1', '-3', '-1', '-3', '4' ],\n             'lR' => [ '-2', '-1', '-3', '0', '-4', '4', '-4', '-3' ],\n             'lI' => [ '0', '-3', '0', '-4', '6', '-4', '-2', '-2' ],\n             'lG' => [ '-4', '-2', '-4', '-2', '-5', '-3', '-1', '-2' ],\n             'pS' => [ '0', '33', '0', '16', '1', '12', '11', '25' ],\n             'pF' => [ '0', '0', '2', '0', '3', '0', '0', '0' ],\n             'pT' => [ '0', '8', '7', '10', '1', '2', '7', '8' ],\n             'pN' => [ '0', '0', '2', '13', '0', '36', '1', '4' ],\n             'pK' => [ '0', '5', '0', '13', '1', '15', '0', '2' ],\n             'pY' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pE' => [ '0', '41', '1', '12', '0', '0', '0', '15' ],\n             'pV' => [ '0', '3', '9', '0', '2', '0', '3', '1' ],\n             'pQ' => [ '0', '0', '0', '15', '0', '4', '0', '3' ],\n             'pM' => [ '100', '0', '66', '0', '2', '0', '0', '0' ],\n             'pC' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pL' => [ '0', '0', '8', '0', '25', '0', '4', '0' ],\n             'pA' => [ '0', '10', '1', '9', '2', '0', '22', '16' ],\n             'pW' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pP' => [ '0', '0', '0', '0', '3', '1', '45', '0' ],\n             'pH' => [ '0', '0', '0', '0', '0', '0', '1', '0' ],\n             'pD' => [ '0', '0', '1', '7', '2', '2', '0', '22' ],\n             'pR' => [ '0', '0', '0', '3', '0', '27', '0', '0' ],\n             'pI' => [ '0', '0', '3', '0', '59', '1', '2', '3' ],\n             'pG' => [ '0', '0', '0', '1', '0', '0', '4', '1' ],\n   );\n\n   my $matrix = Bio::Matrix::PSM::ProtMatrix( %param );\n\n\n   my $site = Bio::Matrix::PSM::ProtMatrix->new(%param);\n   # Or get it from a file:\n   use Bio::Matrix::PSM::IO;\n   my $psmIO = Bio::Matrix::PSM::IO->new(-file => $file, -format => 'psi-blast');\n   while (my $psm = $psmIO->next_psm) {\n      #Now we have a Bio::Matrix::PSM::Psm object, \n      # see Bio::Matrix::PSM::PsmI for details\n      #This is a Bio::Matrix::PSM::ProtMatrix object now\n      my $matrix = $psm->matrix;   \n   }\n\n   # Get a simple consensus, where alphabet is:\n   # {A, R, N, D, C, Q, E, G, H, I, L, K, M, F, P, S, T, W, Y, V,}\n   # choosing the highest probability or N if prob is too low\n   my $consensus = $site->consensus;\n\n   # Retrieving and using regular expressions:\n   my $regexp = $site->regexp;\n   my $count = grep($regexp,$seq);\n   my $count = ($seq=~ s/$regexp/$1/eg);\n   print \"Motif $mid is present $count times in this sequence\\n\";\n\n=head1 DESCRIPTION\n\nProtMatrix is designed to provide some basic methods when working with\nposition scoring (weight) matrices related to protein sequences.  A\nprotein PSM consists of 20 vectors with 20 frequencies (one per amino\nacid per position).  This is the minimum information you should\nprovide to construct a PSM object.  The vectors can be provided as\nstrings with frequencies where the frequency is {0..a} and a=1. This\nis the way MEME compressed representation of a matrix and it is quite\nuseful when working with relational DB.  If arrays are provided as an\ninput (references to arrays actually) they can be any number, real or\ninteger (frequency or count).\n\nWhen creating the object the constructor will check for positions that\nequal 0.  If such is found it will increase the count for all\npositions by one and recalculate the frequency.  Potential bug - if\nyou are using frequencies and one of the positions is 0 it will change\nsignificantly.  However, you should never have frequency that equals\n0.\n\nThrows an exception if: You mix as an input array and string (for\nexample A matrix is given as array, C - as string).  The position\nvector is (0,0,0,0).  One of the probability vectors is shorter than\nthe rest.\n\nSummary of the methods I use most frequently (details bellow):\n\n   iupac - return IUPAC compliant consensus as a string\n   score - Returns the score as a real number\n   IC - information content. Returns a real number\n   id - identifier. Returns a string\n   accession - accession number. Returns a string\n   next_pos - return the sequence probably for each letter, IUPAC\n         symbol, IUPAC probability and simple sequence\n   consenus letter for this position. Rewind at the end. Returns a hash.\n   pos - current position get/set. Returns an integer.\n   regexp - construct a regular expression based on IUPAC consensus.\n         For example AGWV will be [Aa][Gg][AaTt][AaCcGg]\n   width - site width\n   get_string - gets the probability vector for a single base as a string.\n   get_array - gets the probability vector for a single base as an array.\n   get_logs_array - gets the log-odds vector for a single base as an array.\n\nNew methods, which might be of interest to anyone who wants to store\nPSM in a relational database without creating an entry for each\nposition is the ability to compress the PSM vector into a string with\nlosing usually less than 1% of the data.  this can be done with:\n\n   my $str=$matrix->get_compressed_freq('A');\nor\n\n   my $str=$matrix->get_compressed_logs('A');\n\nLoading from a database should be done with new, but is not yet implemented.\nHowever you can still uncompress such string with:\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1,1); for PSM\n\nor\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1000,2); for log odds\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to one\nof the Bioperl mailing lists.   Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nthe bugs and their resolution.  Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 AUTHOR - James Thompson\n\nEmail tex@biosysadmin.com\n\n=head1 APPENDIX\n\n\n\n# Let the code begin...\npackage Bio::Matrix::PSM::ProtMatrix;\nuse strict;\n\nuse base qw(Bio::Root::Root Bio::Matrix::PSM::SiteMatrixI);\n\n=head2 new\n\n Title    : new\n Usage    : my $site = Bio::Matrix::PSM::ProtMatrix->new( \n               %probs,\n               %logs,\n               -IC    => $ic,\n               -e_val => $score, \n               -id    => $mid\n               -model => \\%model\n            );\n Function : Creates a new Bio::Matrix::PSM::ProtMatrix object from memory\n Throws   : If inconsistent data for all vectors (all 20 amino acids) is\n               provided, if you mix input types (string vs array) or if a\n               position freq is 0.\n Example  :\n Returns  : Bio::Matrix::PSM::ProtMatrix object\n Args     : Hash references to log-odds scores and probabilities for\n            position-specific scoring info, e-value (optional), information\n            content (optional), id (optional), model for background distribution\n            of proteins (optional).\n\n\nsub new {\n   my ($class, @args) = @_;\n   my $self = $class->SUPER::new(@args);\n   my $consensus;\n   #Too many things to rearrange, and I am creating simultanuously >500 \n   # such objects routinely, so this becomes performance issue\n   my %input;\n   while( @args ) {\n     (my $key = shift @args) =~ s/-//gi; #deletes all dashes (only dashes)!\n     $input{$key} = shift @args;\n   }\n\n   # get a protein alphabet for processing log-odds scores and probabilities\n   # maybe change this later on to allow for non-standard aa lists? \n   my @alphabet = qw/A  R  N  D  C  Q  E  G  H  I  L  K  M  F  P  S  T  W  Y  V/;\n\n   foreach my $aa (@alphabet) {\n      $self->{\"log$aa\"}  = defined($input{\"l$aa\"}) ? $input{\"l$aa\"} \n         : $self->throw(\"Error: No log-odds information for $aa!\");\n      $self->{\"prob$aa\"}  = defined($input{\"p$aa\"}) ? $input{\"p$aa\"} \n         : $self->throw(\"Error: No probability information for $aa!\");\n   }\n   \n   $self->{_position} = 0;\n   $self->{IC}        = $input{IC};\n   $self->{e_val}     = $input{e_val};\n   $self->{sites}     = $input{sites};\n   $self->{width}     = $input{width};\n   $self->{accession_number} = $input{accession_number};\n   $self->{_correction} = defined($input{correction}) ?\n      $input{correction} : 1 ; # Correction might be unwanted- supply your own\n   # No id provided, null for the sake of rel db\n   $self->{id} = defined($input{id}) ? $input{id} : 'null'; \n   $self->{_alphabet} = \\@alphabet;\n\n   #Make consensus, throw if any one of the vectors is shorter\n   $self = _calculate_consensus($self,$input{model});\n   return $self;\n}\n\n=head2 alphabet\n\n Title    : Returns an array (or array reference if desired) to the alphabet \n Usage    :\n Function : Returns an array (or array reference) containing all of the\n            allowable characters for this matrix.\n Throws   :\n Example  :\n Returns  : Array or arrary reference.\n Args     :\n\n\nsub alphabet {\n   my $self = shift;\n   if ( wantarray ) {\n      return $self->{_alphabet};\n   } else {\n      return @{$self->{_alphabet}};\n   }\n}\n\n=head2 _calculate_consensus\n\n Title    : _calculate_consensus\n Usage    :\n Function : Calculates the consensus sequence for this matrix. \n Throws   :\n Example  :\n Returns  :\n Args     :\n\n\nsub _calculate_consensus {\n   my $self   = shift;\n   my $thresh = shift;\n   \n   # verify that all of the array lengths in %probs are the same\n   my @lengths = map { scalar(@$_) } map {$self->{\"prob$_\"}} @{ $self->{_alphabet} };\n   my $len = shift @lengths;\n   for ( @lengths ) {\n      if ( $_ ne $len ) { $self->throw( \"Probability matrix is damaged!\\n\" ) };\n   }\n\n   # iterate over probs, generate the most likely sequence and put it into \n   # $self->{seq}. Put the probability of this sequence into $self->{seqp}.\n   for ( my $i = 0; $i < $len; $i++ ) {\n      # get a list of all the probabilities at position $i, ordered by $self->{_alphabet}\n      my @probs = map { ${$self->{\"prob$_\"}}[$i] } @{ $self->{_alphabet} };\n      # calculate the consensus of @probs, put sequence into seqp and probabilities into seqp\n      (${$self->{seq}}[$i],${$self->{seqp}}[$i]) = $self->_to_cons( @probs, $thresh );\n   }\n\n   return $self;\n}\n\n=head2 next_pos\n\n Title    : next_pos\n Usage    :\n Function : Retrives the next position features: frequencies for all 20 amino\n            acids, log-odds scores for all 20 amino acids at this position,\n            the main (consensus) letter at this position, the probability\n            for the consensus letter to occur at this position and the relative\n            current position as an integer.\n Throws   :\n Example  :\n Returns  : hash (or hash reference) (pA,pR,pN,pD,...,logA,logR,logN,logD,aa,prob,rel)\n            - pN entries represent the probability for amino acid N\n            to be at this position\n            - logN entries represent the log-odds score for having amino acid\n            N at this position\n            - aa is the consensus amino acid\n            - prob is the probability for the consensus amino acid to be at this \n            position\n            - rel is the relative index of the current position (integer)\n Args      : none\n\n\n\nsub next_pos {\n      my $self = shift;\n         $self->throw(\"instance method called on class\") unless ref $self;\n\n      my $len = @{$self->{seq}};\n      my $pos = $self->{_position};\n\n      # return a PSM if we're still within range\n      if ($pos<$len) {\n\n         my %probs = map { (\"p$_\", ${$self->{\"prob$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my %logs  = map { (\"l$_\", ${$self->{\"log$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my $base  = ${$self->{seq}}[$pos];\n         my $prob  = ${$self->{seqp}}[$pos];\n\n         $self->{_position}++;\n         my %hash = ( %probs, %logs, base => $base, rel => $pos, prob => $prob );\n         \n         # decide whether to return the hash or a reference to it\n         if ( wantarray ) {\n            return %hash;\n         } else {\n            return \\%hash;\n         }\n   } else { # otherwise, reset $self->{_position} and return nothing\n      $self->{_position} = 0; \n      return;\n   }\n}\n\n\n=head2 curpos\n\n Title    : curpos\n Usage    :\n Function : Gets/sets the current position. \n Throws   :\n Example  :\n Returns  : Current position (integer).\n Args     : New position (integer).\n\n\nsub curpos {\n      my $self = shift;\n      if (@_) { $self->{_position} = shift; }\n      return $self->{_position};\n}\n\n\n=head2 e_val\n\n Title    : e_val\n Usage    :\n Function : Gets/sets the e-value\n Throws   :\n Example  :\n Returns  : \n Args     : real number\n\n\nsub e_val {\n      my $self = shift;\n      if (@_) { $self->{e_val} = shift; }\n      return $self->{e_val};\n}\n\n\n=head2 IC\n\n Title    : IC\n Usage    :\n Function : Position-specific information content.\n Throws   :\n Example  :\n Returns  : Information content for current position.\n Args     : Information content for current position.\n\n\nsub IC {\n      my $self = shift;\n      if (@_) { $self->{IC} = shift; }\n      return $self->{IC};\n}\n\n=head2 accession_number\n\n Title    : accession_number\n Usage    :\n Function: accession number, this will be unique id for the ProtMatrix object as\n            well for any other object, inheriting from ProtMatrix.\n Throws   :\n Example  :\n Returns  : New accession number (string)\n Args     : Accession number (string)\n\n\nsub accession_number {\n      my $self = shift;\n      if (@_) { $self->{accession_number} = shift; }\n      return $self->{accession_number};\n}\n\n=head2 consensus\n\n Title    : consensus\n Usage    :\n Function : Returns the consensus sequence for this PSM.\n Throws   : if supplied with thresold outisde 5..10 range\n Example  :\n Returns  : string\n Args     : (optional) threshold value 5 to 10 (corresponds to 50-100% at each position\n\n\nsub consensus {\n   my $self = shift;\n   my $thresh=shift;\n   $self->_calculate_consensus($thresh) if ($thresh); #Change of threshold\n   my $consensus='';\n\n   foreach my $letter (@{$self->{seq}}) {\n       $consensus .= $letter;\n   }\n\n   return $consensus;\n}\n\nsub IUPAC {\n   my $self = shift;\n   return $self->consensus;\n}\n\n\n=head2 get_string\n\n Title   : get_string\n Usage   :\n Function: Returns given probability vector as a string. Useful if you want to\n            store things in a rel database, where arrays are not first choice\n Throws  : If the argument is outside {A,C,G,T}\n Example :\n Returns : string\n Args    : character {A,C,G,T}\n\n\nsub get_string {\n   my $self = shift;\n   my $base = shift;\n   my $string = '';\n\n   my @prob = @{$self->{\"prob$base\"}};\n   if ( ! @prob ) {\n      $self->throw( \"No such base: $base\\n\");\n   }\n\n   foreach my $prob (@prob) {\n      my $corrected = $prob*10;\n      my $next = sprintf(\"%.0f\",$corrected);\n      $next = 'a' if ($next eq '10');\n      $string .= $next;\n   }\n   return $string;\n}\n\n\n\n=head2 width\n\n Title    : width\n Usage    :\n Function : Returns the length of the site\n Throws   :\n Example  :\n Returns  : number\n Args     :\n\n\nsub width {\n   my $self = shift;\n   my $width = @{$self->{probA}};\n   return $width;\n}\n\n=head2 get_array\n\n Title    : get_array\n Usage    :\n Function : Returns an array with frequencies for a specified amino acid.\n Throws   :\n Example  :\n Returns  : Array representing frequencies for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"prob$letter\"}}; \n}\n\n\n=head2 get_logs_array\n\n Title    : get_logs_array\n Usage    :\n Function : Returns an array with log_odds for a specified base\n Throws   :\n Example  :\n Returns  : Array representing log-odds scores for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_logs_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"log$letter\"}}; \n}\n\n=head2 id\n\n Title    : id\n Usage    :\n Function : Gets/sets the site id\n Throws   :\n Example  :\n Returns  : string\n Args     : string\n\n\nsub id {\n      my $self = shift;\n      if (@_) { $self->{id} = shift; }\n      return $self->{id};\n}\n\n=head2 regexp\n\n Title    : regexp\n Usage    :\n Function : Returns a case-insensitive regular expression which matches the\n            IUPAC convention.  X's in consensus sequence will match anything.     \n Throws   :\n Example  :\n Returns  : string\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n\n\nsub regexp {\n   my $self = shift;\n   my $threshold = 20;\n   if ( @_ ) { my $threshold = shift };\n\n   my @alphabet = @{$self->{_alphabet}};\n   my $width = $self->width;\n   my (@regexp, $i);\n   for ( $i = 0; $i < $width; $i++ ) {\n      # get an array of the residues at this position with p > $threshold\n      my @letters = map { uc($_).lc($_) } grep { $self->{\"prob$_\"}->[$i] >= $threshold } @alphabet;\n\n      my $reg;\n      if ( scalar(@letters) == 0 ) {\n         $reg = '\\.';\n      } else {\n         $reg = '['.join('',@letters).']';\n      }\n      push @regexp, $reg;\n   }\n\n   if ( wantarray ) { \n      return @regexp;\n   } else {\n      return join '', @regexp;\n   }\n}\n\n\n=head2 regexp_array\n\n Title    : regexp_array\n Usage    :\n Function : Returns an array of position-specific regular expressions.\n             X's in consensus sequence will match anything.      \n Throws   :\n Example  :\n Returns  : Array of position-specific regular expressions.\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n Notes    : Simply calls regexp method in list context.\n\n\nsub regexp_array {\n   my $self = shift;\n   \n   return @{ $self->regexp };\n}\n\n\n=head2 _compress_array\n\n Title    : _compress_array\n Usage    :\n Function :  Will compress an array of real signed numbers to a string (ie vector of bytes)\n             -127 to +127 for bi-directional(signed) and 0..255 for unsigned ;\n Throws   :\n Example  :  Internal stuff\n Returns  :  String\n Args     :  array reference, followed by max value and direction (optional, defaults to 1),\n             direction of 1 is unsigned, anything else is signed. \n\n\nsub _compress_array {\n   my ($array,$lm,$direct)=@_;\n   my $str;\n   return unless(($array) && ($lm));\n   $direct=1 unless ($direct);\n   my $k1= ($direct==1) ? (255/$lm) : (127/$lm);\n   foreach my $c (@{$array}) {\n      $c=$lm if ($c>$lm);\n      $c=-$lm if (($c<-$lm) && ($direct !=1));\n      $c=0 if (($c<0) && ($direct ==1));\n      my $byte=int($k1*$c);\n      $byte=127+$byte if ($direct !=1);#Clumsy, should be really shift the bits\n      my $char=chr($byte);\n      $str.=$char;\n   }\n   return $str;\n}\n\n=head2 _uncompress_string\n\n Title    : _uncompress_string\n Usage    :\n Function :   Will uncompress a string (vector of bytes) to create an array of real\n                  signed numbers (opposite to_compress_array)\n Throws   :\n Example  :   Internal stuff\n Returns  :   string, followed by max value and direction (optional, defaults to 1),\n              direction of 1 is unsigned, anything else is signed.\n Args     :   array","label":"_uncompress_string($str,$lm,$direct)"},"detail":"($str,$lm,$direct)"},{"containerName":"main::","name":"get_compressed_freq","definition":"sub","range":{"start":{"line":731,"character":0},"end":{"character":9999,"line":744}},"kind":12,"line":731,"children":[{"localvar":"my","name":"$self","definition":"my","containerName":"get_compressed_freq","line":732,"kind":13},{"kind":13,"line":733,"containerName":"get_compressed_freq","definition":"my","name":"$base","localvar":"my"},{"containerName":"get_compressed_freq","definition":"my","name":"$string","localvar":"my","kind":13,"line":734},{"containerName":"get_compressed_freq","definition":"my","name":"@prob","localvar":"my","kind":13,"line":735},{"name":"$base","containerName":"get_compressed_freq","kind":13,"line":737},{"containerName":"get_compressed_freq","name":"@prob","line":738,"kind":13},{"kind":13,"line":738,"containerName":"get_compressed_freq","name":"$self"},{"kind":13,"line":738,"containerName":"get_compressed_freq","name":"$self"},{"containerName":"get_compressed_freq","name":"$base","line":741,"kind":13},{"name":"@prob","containerName":"get_compressed_freq","line":742,"kind":13},{"kind":13,"line":742,"containerName":"get_compressed_freq","name":"$self"},{"containerName":"get_compressed_freq","name":"$self","kind":13,"line":742}]},{"kind":12,"line":736,"name":"BASE"},{"kind":12,"line":738,"name":"probA"},{"name":"probA","kind":12,"line":738},{"name":"BASE","kind":12,"line":739},{"line":742,"kind":12,"name":"probG"},{"line":742,"kind":12,"name":"probG"},{"name":"BASE","kind":12,"line":743},{"line":745,"kind":13,"name":"%base","containerName":null},{"line":746,"kind":13,"name":"@prob","containerName":null},{"containerName":null,"name":"%self","kind":13,"line":746},{"line":746,"kind":12,"name":"probC"},{"line":746,"kind":13,"name":"%self","containerName":null},{"line":746,"kind":12,"name":"probC"},{"name":"BASE","line":747,"kind":12},{"containerName":null,"name":"%base","kind":13,"line":749},{"containerName":null,"name":"@prob","kind":13,"line":750},{"kind":13,"line":750,"name":"%self","containerName":null},{"name":"probT","kind":12,"line":750},{"line":750,"kind":13,"name":"%self","containerName":null},{"kind":12,"line":750,"name":"probT"},{"kind":12,"line":751,"name":"BASE"},{"name":"$self","containerName":null,"line":753,"kind":13},{"name":"throw","containerName":"main::","line":753,"kind":12},{"kind":13,"line":755,"containerName":null,"definition":"my","name":"$str","localvar":"my"},{"containerName":null,"name":"@prob","line":755,"kind":13},{"line":756,"kind":13,"name":"$str","containerName":null},{"children":[{"kind":13,"line":774,"containerName":"sequence_match_weight","definition":"my","name":"$self","localvar":"my"},{"kind":13,"line":774,"containerName":"sequence_match_weight","name":"$seq"},{"kind":13,"line":775,"containerName":"sequence_match_weight","name":"$self"},{"containerName":"sequence_match_weight","name":"$seqlen","definition":"my","localvar":"my","kind":13,"line":777},{"kind":13,"line":777,"name":"$seq","containerName":"sequence_match_weight"},{"kind":13,"line":778,"containerName":"sequence_match_weight","definition":"my","name":"$width","localvar":"my"},{"line":778,"kind":13,"name":"$self","containerName":"sequence_match_weight"},{"name":"width","containerName":"sequence_match_weight","kind":12,"line":778},{"containerName":"sequence_match_weight","name":"$self","line":779,"kind":13},{"line":779,"kind":12,"name":"throw","containerName":"sequence_match_weight"},{"containerName":"sequence_match_weight","name":"$seq","kind":13,"line":780},{"containerName":"sequence_match_weight","name":"$self","kind":13,"line":780},{"name":"width","containerName":"sequence_match_weight","line":780,"kind":12},{"kind":13,"line":782,"name":"$score","definition":"my","containerName":"sequence_match_weight","localvar":"my"},{"kind":13,"line":782,"name":"$i","containerName":"sequence_match_weight"},{"localvar":"my","name":"$letter","definition":"my","containerName":"sequence_match_weight","line":783,"kind":13},{"name":"$seq","containerName":"sequence_match_weight","line":783,"kind":13},{"line":785,"kind":13,"containerName":"sequence_match_weight","name":"$score"},{"kind":13,"line":785,"containerName":"sequence_match_weight","name":"$self"},{"containerName":"sequence_match_weight","name":"$i","line":785,"kind":13},{"line":786,"kind":13,"name":"$i","containerName":"sequence_match_weight"},{"kind":13,"line":788,"name":"$score","containerName":"sequence_match_weight"}],"line":773,"kind":12,"signature":{"label":"sequence_match_weight($self,$seq)","parameters":[{"label":"$self"},{"label":"$seq"}],"documentation":"1;\n# $Id: ProtMatrix.pm 16123 2009-09-17 12:57:27Z cjfields $\n#---------------------------------------------------------\n\n=head1 NAME\n\nBio::Matrix::PSM::ProtMatrix - SiteMatrixI implementation, holds a\nposition scoring matrix (or position weight matrix) with log-odds scoring\ninformation.\n\n=head1 SYNOPSIS\n\n   use Bio::Matrix::PSM::ProtMatrix;\n   # Create from memory by supplying probability matrix hash both as strings or\n   # arrays where the frequencies   Hash entries of the form lN refer to an array\n   # of position-specific log-odds scores for amino acid N. Hash entries of the\n   # form pN represent the position-specific probability of finding amino acid N.\n\n   my %param = (\n             'id' => 'A. thaliana protein atp1',\n             '-e_val' => $score,\n             'lS' => [ '-2', '3', '-3', '2', '-3', '1', '1', '3' ],\n             'lF' => [ '-1', '-4', '0', '-5', '0', '-5', '-4', '-4' ],\n             'lT' => [ '-1', '1', '0', '1', '-2', '-1', '0', '1' ],\n             'lN' => [ '-3', '-1', '-2', '3', '-5', '5', '-2', '0' ],\n             'lK' => [ '-2', '0', '-3', '2', '-3', '2', '-3', '-1' ],\n             'lY' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-4', '-4' ],\n             'lE' => [ '-3', '4', '-3', '2', '-4', '-2', '-3', '2' ],\n             'lV' => [ '0', '-2', '1', '-4', '1', '-4', '-1', '-3' ],\n             'lQ' => [ '-1', '0', '-2', '3', '-4', '1', '-3', '0' ],\n             'lM' => [ '8', '-3', '8', '-3', '1', '-3', '-3', '-3' ],\n             'lC' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-3', '-3' ],\n             'lL' => [ '1', '-3', '1', '-4', '3', '-4', '-2', '-4' ],\n             'lA' => [ '-2', '1', '-2', '0', '-2', '-2', '2', '2' ],\n             'lW' => [ '-2', '-4', '-3', '-5', '-4', '-5', '-5', '-5' ],\n             'lP' => [ '-3', '-2', '-4', '-3', '-1', '-3', '6', '-3' ],\n             'lH' => [ '-2', '-2', '-3', '-2', '-5', '-2', '-2', '-3' ],\n             'lD' => [ '-4', '-1', '-3', '1', '-3', '-1', '-3', '4' ],\n             'lR' => [ '-2', '-1', '-3', '0', '-4', '4', '-4', '-3' ],\n             'lI' => [ '0', '-3', '0', '-4', '6', '-4', '-2', '-2' ],\n             'lG' => [ '-4', '-2', '-4', '-2', '-5', '-3', '-1', '-2' ],\n             'pS' => [ '0', '33', '0', '16', '1', '12', '11', '25' ],\n             'pF' => [ '0', '0', '2', '0', '3', '0', '0', '0' ],\n             'pT' => [ '0', '8', '7', '10', '1', '2', '7', '8' ],\n             'pN' => [ '0', '0', '2', '13', '0', '36', '1', '4' ],\n             'pK' => [ '0', '5', '0', '13', '1', '15', '0', '2' ],\n             'pY' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pE' => [ '0', '41', '1', '12', '0', '0', '0', '15' ],\n             'pV' => [ '0', '3', '9', '0', '2', '0', '3', '1' ],\n             'pQ' => [ '0', '0', '0', '15', '0', '4', '0', '3' ],\n             'pM' => [ '100', '0', '66', '0', '2', '0', '0', '0' ],\n             'pC' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pL' => [ '0', '0', '8', '0', '25', '0', '4', '0' ],\n             'pA' => [ '0', '10', '1', '9', '2', '0', '22', '16' ],\n             'pW' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pP' => [ '0', '0', '0', '0', '3', '1', '45', '0' ],\n             'pH' => [ '0', '0', '0', '0', '0', '0', '1', '0' ],\n             'pD' => [ '0', '0', '1', '7', '2', '2', '0', '22' ],\n             'pR' => [ '0', '0', '0', '3', '0', '27', '0', '0' ],\n             'pI' => [ '0', '0', '3', '0', '59', '1', '2', '3' ],\n             'pG' => [ '0', '0', '0', '1', '0', '0', '4', '1' ],\n   );\n\n   my $matrix = Bio::Matrix::PSM::ProtMatrix( %param );\n\n\n   my $site = Bio::Matrix::PSM::ProtMatrix->new(%param);\n   # Or get it from a file:\n   use Bio::Matrix::PSM::IO;\n   my $psmIO = Bio::Matrix::PSM::IO->new(-file => $file, -format => 'psi-blast');\n   while (my $psm = $psmIO->next_psm) {\n      #Now we have a Bio::Matrix::PSM::Psm object, \n      # see Bio::Matrix::PSM::PsmI for details\n      #This is a Bio::Matrix::PSM::ProtMatrix object now\n      my $matrix = $psm->matrix;   \n   }\n\n   # Get a simple consensus, where alphabet is:\n   # {A, R, N, D, C, Q, E, G, H, I, L, K, M, F, P, S, T, W, Y, V,}\n   # choosing the highest probability or N if prob is too low\n   my $consensus = $site->consensus;\n\n   # Retrieving and using regular expressions:\n   my $regexp = $site->regexp;\n   my $count = grep($regexp,$seq);\n   my $count = ($seq=~ s/$regexp/$1/eg);\n   print \"Motif $mid is present $count times in this sequence\\n\";\n\n=head1 DESCRIPTION\n\nProtMatrix is designed to provide some basic methods when working with\nposition scoring (weight) matrices related to protein sequences.  A\nprotein PSM consists of 20 vectors with 20 frequencies (one per amino\nacid per position).  This is the minimum information you should\nprovide to construct a PSM object.  The vectors can be provided as\nstrings with frequencies where the frequency is {0..a} and a=1. This\nis the way MEME compressed representation of a matrix and it is quite\nuseful when working with relational DB.  If arrays are provided as an\ninput (references to arrays actually) they can be any number, real or\ninteger (frequency or count).\n\nWhen creating the object the constructor will check for positions that\nequal 0.  If such is found it will increase the count for all\npositions by one and recalculate the frequency.  Potential bug - if\nyou are using frequencies and one of the positions is 0 it will change\nsignificantly.  However, you should never have frequency that equals\n0.\n\nThrows an exception if: You mix as an input array and string (for\nexample A matrix is given as array, C - as string).  The position\nvector is (0,0,0,0).  One of the probability vectors is shorter than\nthe rest.\n\nSummary of the methods I use most frequently (details bellow):\n\n   iupac - return IUPAC compliant consensus as a string\n   score - Returns the score as a real number\n   IC - information content. Returns a real number\n   id - identifier. Returns a string\n   accession - accession number. Returns a string\n   next_pos - return the sequence probably for each letter, IUPAC\n         symbol, IUPAC probability and simple sequence\n   consenus letter for this position. Rewind at the end. Returns a hash.\n   pos - current position get/set. Returns an integer.\n   regexp - construct a regular expression based on IUPAC consensus.\n         For example AGWV will be [Aa][Gg][AaTt][AaCcGg]\n   width - site width\n   get_string - gets the probability vector for a single base as a string.\n   get_array - gets the probability vector for a single base as an array.\n   get_logs_array - gets the log-odds vector for a single base as an array.\n\nNew methods, which might be of interest to anyone who wants to store\nPSM in a relational database without creating an entry for each\nposition is the ability to compress the PSM vector into a string with\nlosing usually less than 1% of the data.  this can be done with:\n\n   my $str=$matrix->get_compressed_freq('A');\nor\n\n   my $str=$matrix->get_compressed_logs('A');\n\nLoading from a database should be done with new, but is not yet implemented.\nHowever you can still uncompress such string with:\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1,1); for PSM\n\nor\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1000,2); for log odds\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to one\nof the Bioperl mailing lists.   Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nthe bugs and their resolution.  Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 AUTHOR - James Thompson\n\nEmail tex@biosysadmin.com\n\n=head1 APPENDIX\n\n\n\n# Let the code begin...\npackage Bio::Matrix::PSM::ProtMatrix;\nuse strict;\n\nuse base qw(Bio::Root::Root Bio::Matrix::PSM::SiteMatrixI);\n\n=head2 new\n\n Title    : new\n Usage    : my $site = Bio::Matrix::PSM::ProtMatrix->new( \n               %probs,\n               %logs,\n               -IC    => $ic,\n               -e_val => $score, \n               -id    => $mid\n               -model => \\%model\n            );\n Function : Creates a new Bio::Matrix::PSM::ProtMatrix object from memory\n Throws   : If inconsistent data for all vectors (all 20 amino acids) is\n               provided, if you mix input types (string vs array) or if a\n               position freq is 0.\n Example  :\n Returns  : Bio::Matrix::PSM::ProtMatrix object\n Args     : Hash references to log-odds scores and probabilities for\n            position-specific scoring info, e-value (optional), information\n            content (optional), id (optional), model for background distribution\n            of proteins (optional).\n\n\nsub new {\n   my ($class, @args) = @_;\n   my $self = $class->SUPER::new(@args);\n   my $consensus;\n   #Too many things to rearrange, and I am creating simultanuously >500 \n   # such objects routinely, so this becomes performance issue\n   my %input;\n   while( @args ) {\n     (my $key = shift @args) =~ s/-//gi; #deletes all dashes (only dashes)!\n     $input{$key} = shift @args;\n   }\n\n   # get a protein alphabet for processing log-odds scores and probabilities\n   # maybe change this later on to allow for non-standard aa lists? \n   my @alphabet = qw/A  R  N  D  C  Q  E  G  H  I  L  K  M  F  P  S  T  W  Y  V/;\n\n   foreach my $aa (@alphabet) {\n      $self->{\"log$aa\"}  = defined($input{\"l$aa\"}) ? $input{\"l$aa\"} \n         : $self->throw(\"Error: No log-odds information for $aa!\");\n      $self->{\"prob$aa\"}  = defined($input{\"p$aa\"}) ? $input{\"p$aa\"} \n         : $self->throw(\"Error: No probability information for $aa!\");\n   }\n   \n   $self->{_position} = 0;\n   $self->{IC}        = $input{IC};\n   $self->{e_val}     = $input{e_val};\n   $self->{sites}     = $input{sites};\n   $self->{width}     = $input{width};\n   $self->{accession_number} = $input{accession_number};\n   $self->{_correction} = defined($input{correction}) ?\n      $input{correction} : 1 ; # Correction might be unwanted- supply your own\n   # No id provided, null for the sake of rel db\n   $self->{id} = defined($input{id}) ? $input{id} : 'null'; \n   $self->{_alphabet} = \\@alphabet;\n\n   #Make consensus, throw if any one of the vectors is shorter\n   $self = _calculate_consensus($self,$input{model});\n   return $self;\n}\n\n=head2 alphabet\n\n Title    : Returns an array (or array reference if desired) to the alphabet \n Usage    :\n Function : Returns an array (or array reference) containing all of the\n            allowable characters for this matrix.\n Throws   :\n Example  :\n Returns  : Array or arrary reference.\n Args     :\n\n\nsub alphabet {\n   my $self = shift;\n   if ( wantarray ) {\n      return $self->{_alphabet};\n   } else {\n      return @{$self->{_alphabet}};\n   }\n}\n\n=head2 _calculate_consensus\n\n Title    : _calculate_consensus\n Usage    :\n Function : Calculates the consensus sequence for this matrix. \n Throws   :\n Example  :\n Returns  :\n Args     :\n\n\nsub _calculate_consensus {\n   my $self   = shift;\n   my $thresh = shift;\n   \n   # verify that all of the array lengths in %probs are the same\n   my @lengths = map { scalar(@$_) } map {$self->{\"prob$_\"}} @{ $self->{_alphabet} };\n   my $len = shift @lengths;\n   for ( @lengths ) {\n      if ( $_ ne $len ) { $self->throw( \"Probability matrix is damaged!\\n\" ) };\n   }\n\n   # iterate over probs, generate the most likely sequence and put it into \n   # $self->{seq}. Put the probability of this sequence into $self->{seqp}.\n   for ( my $i = 0; $i < $len; $i++ ) {\n      # get a list of all the probabilities at position $i, ordered by $self->{_alphabet}\n      my @probs = map { ${$self->{\"prob$_\"}}[$i] } @{ $self->{_alphabet} };\n      # calculate the consensus of @probs, put sequence into seqp and probabilities into seqp\n      (${$self->{seq}}[$i],${$self->{seqp}}[$i]) = $self->_to_cons( @probs, $thresh );\n   }\n\n   return $self;\n}\n\n=head2 next_pos\n\n Title    : next_pos\n Usage    :\n Function : Retrives the next position features: frequencies for all 20 amino\n            acids, log-odds scores for all 20 amino acids at this position,\n            the main (consensus) letter at this position, the probability\n            for the consensus letter to occur at this position and the relative\n            current position as an integer.\n Throws   :\n Example  :\n Returns  : hash (or hash reference) (pA,pR,pN,pD,...,logA,logR,logN,logD,aa,prob,rel)\n            - pN entries represent the probability for amino acid N\n            to be at this position\n            - logN entries represent the log-odds score for having amino acid\n            N at this position\n            - aa is the consensus amino acid\n            - prob is the probability for the consensus amino acid to be at this \n            position\n            - rel is the relative index of the current position (integer)\n Args      : none\n\n\n\nsub next_pos {\n      my $self = shift;\n         $self->throw(\"instance method called on class\") unless ref $self;\n\n      my $len = @{$self->{seq}};\n      my $pos = $self->{_position};\n\n      # return a PSM if we're still within range\n      if ($pos<$len) {\n\n         my %probs = map { (\"p$_\", ${$self->{\"prob$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my %logs  = map { (\"l$_\", ${$self->{\"log$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my $base  = ${$self->{seq}}[$pos];\n         my $prob  = ${$self->{seqp}}[$pos];\n\n         $self->{_position}++;\n         my %hash = ( %probs, %logs, base => $base, rel => $pos, prob => $prob );\n         \n         # decide whether to return the hash or a reference to it\n         if ( wantarray ) {\n            return %hash;\n         } else {\n            return \\%hash;\n         }\n   } else { # otherwise, reset $self->{_position} and return nothing\n      $self->{_position} = 0; \n      return;\n   }\n}\n\n\n=head2 curpos\n\n Title    : curpos\n Usage    :\n Function : Gets/sets the current position. \n Throws   :\n Example  :\n Returns  : Current position (integer).\n Args     : New position (integer).\n\n\nsub curpos {\n      my $self = shift;\n      if (@_) { $self->{_position} = shift; }\n      return $self->{_position};\n}\n\n\n=head2 e_val\n\n Title    : e_val\n Usage    :\n Function : Gets/sets the e-value\n Throws   :\n Example  :\n Returns  : \n Args     : real number\n\n\nsub e_val {\n      my $self = shift;\n      if (@_) { $self->{e_val} = shift; }\n      return $self->{e_val};\n}\n\n\n=head2 IC\n\n Title    : IC\n Usage    :\n Function : Position-specific information content.\n Throws   :\n Example  :\n Returns  : Information content for current position.\n Args     : Information content for current position.\n\n\nsub IC {\n      my $self = shift;\n      if (@_) { $self->{IC} = shift; }\n      return $self->{IC};\n}\n\n=head2 accession_number\n\n Title    : accession_number\n Usage    :\n Function: accession number, this will be unique id for the ProtMatrix object as\n            well for any other object, inheriting from ProtMatrix.\n Throws   :\n Example  :\n Returns  : New accession number (string)\n Args     : Accession number (string)\n\n\nsub accession_number {\n      my $self = shift;\n      if (@_) { $self->{accession_number} = shift; }\n      return $self->{accession_number};\n}\n\n=head2 consensus\n\n Title    : consensus\n Usage    :\n Function : Returns the consensus sequence for this PSM.\n Throws   : if supplied with thresold outisde 5..10 range\n Example  :\n Returns  : string\n Args     : (optional) threshold value 5 to 10 (corresponds to 50-100% at each position\n\n\nsub consensus {\n   my $self = shift;\n   my $thresh=shift;\n   $self->_calculate_consensus($thresh) if ($thresh); #Change of threshold\n   my $consensus='';\n\n   foreach my $letter (@{$self->{seq}}) {\n       $consensus .= $letter;\n   }\n\n   return $consensus;\n}\n\nsub IUPAC {\n   my $self = shift;\n   return $self->consensus;\n}\n\n\n=head2 get_string\n\n Title   : get_string\n Usage   :\n Function: Returns given probability vector as a string. Useful if you want to\n            store things in a rel database, where arrays are not first choice\n Throws  : If the argument is outside {A,C,G,T}\n Example :\n Returns : string\n Args    : character {A,C,G,T}\n\n\nsub get_string {\n   my $self = shift;\n   my $base = shift;\n   my $string = '';\n\n   my @prob = @{$self->{\"prob$base\"}};\n   if ( ! @prob ) {\n      $self->throw( \"No such base: $base\\n\");\n   }\n\n   foreach my $prob (@prob) {\n      my $corrected = $prob*10;\n      my $next = sprintf(\"%.0f\",$corrected);\n      $next = 'a' if ($next eq '10');\n      $string .= $next;\n   }\n   return $string;\n}\n\n\n\n=head2 width\n\n Title    : width\n Usage    :\n Function : Returns the length of the site\n Throws   :\n Example  :\n Returns  : number\n Args     :\n\n\nsub width {\n   my $self = shift;\n   my $width = @{$self->{probA}};\n   return $width;\n}\n\n=head2 get_array\n\n Title    : get_array\n Usage    :\n Function : Returns an array with frequencies for a specified amino acid.\n Throws   :\n Example  :\n Returns  : Array representing frequencies for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"prob$letter\"}}; \n}\n\n\n=head2 get_logs_array\n\n Title    : get_logs_array\n Usage    :\n Function : Returns an array with log_odds for a specified base\n Throws   :\n Example  :\n Returns  : Array representing log-odds scores for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_logs_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"log$letter\"}}; \n}\n\n=head2 id\n\n Title    : id\n Usage    :\n Function : Gets/sets the site id\n Throws   :\n Example  :\n Returns  : string\n Args     : string\n\n\nsub id {\n      my $self = shift;\n      if (@_) { $self->{id} = shift; }\n      return $self->{id};\n}\n\n=head2 regexp\n\n Title    : regexp\n Usage    :\n Function : Returns a case-insensitive regular expression which matches the\n            IUPAC convention.  X's in consensus sequence will match anything.     \n Throws   :\n Example  :\n Returns  : string\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n\n\nsub regexp {\n   my $self = shift;\n   my $threshold = 20;\n   if ( @_ ) { my $threshold = shift };\n\n   my @alphabet = @{$self->{_alphabet}};\n   my $width = $self->width;\n   my (@regexp, $i);\n   for ( $i = 0; $i < $width; $i++ ) {\n      # get an array of the residues at this position with p > $threshold\n      my @letters = map { uc($_).lc($_) } grep { $self->{\"prob$_\"}->[$i] >= $threshold } @alphabet;\n\n      my $reg;\n      if ( scalar(@letters) == 0 ) {\n         $reg = '\\.';\n      } else {\n         $reg = '['.join('',@letters).']';\n      }\n      push @regexp, $reg;\n   }\n\n   if ( wantarray ) { \n      return @regexp;\n   } else {\n      return join '', @regexp;\n   }\n}\n\n\n=head2 regexp_array\n\n Title    : regexp_array\n Usage    :\n Function : Returns an array of position-specific regular expressions.\n             X's in consensus sequence will match anything.      \n Throws   :\n Example  :\n Returns  : Array of position-specific regular expressions.\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n Notes    : Simply calls regexp method in list context.\n\n\nsub regexp_array {\n   my $self = shift;\n   \n   return @{ $self->regexp };\n}\n\n\n=head2 _compress_array\n\n Title    : _compress_array\n Usage    :\n Function :  Will compress an array of real signed numbers to a string (ie vector of bytes)\n             -127 to +127 for bi-directional(signed) and 0..255 for unsigned ;\n Throws   :\n Example  :  Internal stuff\n Returns  :  String\n Args     :  array reference, followed by max value and direction (optional, defaults to 1),\n             direction of 1 is unsigned, anything else is signed. \n\n\nsub _compress_array {\n   my ($array,$lm,$direct)=@_;\n   my $str;\n   return unless(($array) && ($lm));\n   $direct=1 unless ($direct);\n   my $k1= ($direct==1) ? (255/$lm) : (127/$lm);\n   foreach my $c (@{$array}) {\n      $c=$lm if ($c>$lm);\n      $c=-$lm if (($c<-$lm) && ($direct !=1));\n      $c=0 if (($c<0) && ($direct ==1));\n      my $byte=int($k1*$c);\n      $byte=127+$byte if ($direct !=1);#Clumsy, should be really shift the bits\n      my $char=chr($byte);\n      $str.=$char;\n   }\n   return $str;\n}\n\n=head2 _uncompress_string\n\n Title    : _uncompress_string\n Usage    :\n Function :   Will uncompress a string (vector of bytes) to create an array of real\n                  signed numbers (opposite to_compress_array)\n Throws   :\n Example  :   Internal stuff\n Returns  :   string, followed by max value and direction (optional, defaults to 1),\n              direction of 1 is unsigned, anything else is signed.\n Args     :   array\n\n\nsub _uncompress_string {\n   my ($str,$lm,$direct)=@_;\n   my @array;\n   return unless(($str) && ($lm));\n   $direct=1 unless ($direct);\n   my $k1= ($direct==1) ? (255/$lm) : (127/$lm);\n   while (my $c=chop($str)) {\n      my $byte=ord($c);\n      $byte=$byte-127 if ($direct !=1);#Clumsy, should be really shift the bits\n      my $num=$byte/$k1;\n      unshift @array,$num;\n   }\n\n   return @array;\n}\n\n=head2 get_compressed_freq\n\n Title    : get_compressed_freq\n Usage    :\n Function:   A method to provide a compressed frequency vector. It uses one byte to\n             code the frequence for one of the probability vectors for one position.\n             Useful for relational database. Improvment of the previous 0..a coding.\n Throws   :\n Example  :   my $strA=$self->get_compressed_freq('A');\n Returns  :   String\n Args     :   char \n\n\nsub get_compressed_freq {\n   my $self=shift;\n   my $base=shift;\n   my $string='';\n   my @prob;\n   BASE: {\n      if ($base eq 'A') {\n         @prob = @{$self->{probA}} unless (!defined($self->{probA}));\n         last BASE;\n      }\n         if ($base eq 'G') {\n         @prob = @{$self->{probG}} unless (!defined($self->{probG}));\n         last BASE;\n      }\n         if ($base eq 'C') {\n         @prob = @{$self->{probC}} unless (!defined($self->{probC}));\n         last BASE;\n      }\n         if ($base eq 'T') {\n         @prob = @{$self->{probT}} unless (!defined($self->{probT}));\n         last BASE;\n      }\n      $self->throw (\"No such base: $base!\\n\");\n   }\n   my $str= _compress_array(\\@prob,1,1);\n   return $str;\n}\n\n=head2 sequence_match_weight\n\n Title    : sequence_match_weight\n Usage    :\n Function :   This method will calculate the score of a match, based on the PSM\n              if such is associated with the matrix object. Returns undef if no\n              PSM data is available.\n Throws   :   if the length of the sequence is different from the matrix width\n Example  :   my $score=$matrix->sequence_match_weight('ACGGATAG');\n Returns  :   Floating point\n Args     :   string"},"detail":"($self,$seq)","definition":"sub","containerName":"main::","range":{"start":{"line":773,"character":0},"end":{"line":789,"character":9999}},"name":"sequence_match_weight"},{"kind":12,"line":775,"name":"logA"},{"range":{"start":{"line":806,"character":0},"end":{"character":9999,"line":814}},"name":"_to_IUPAC","line":806,"children":[{"kind":13,"line":807,"containerName":"_to_IUPAC","name":"$self","definition":"my","localvar":"my"},{"line":807,"kind":13,"containerName":"_to_IUPAC","name":"@probs"},{"kind":13,"line":807,"name":"$thresh","containerName":"_to_IUPAC"},{"kind":13,"line":811,"containerName":"_to_IUPAC","name":"$thresh"},{"name":"$thresh","containerName":"_to_IUPAC","kind":13,"line":811},{"kind":13,"line":813,"containerName":"_to_IUPAC","definition":"my","name":"$IUPAC_aa","localvar":"my"},{"name":"$max_prob","containerName":"_to_IUPAC","kind":13,"line":813},{"line":813,"kind":13,"containerName":"_to_IUPAC","name":"$thresh"},{"kind":13,"line":814,"containerName":"_to_IUPAC","definition":"my","name":"$aa","localvar":"my"},{"containerName":"_to_IUPAC","name":"$self","kind":13,"line":814}],"kind":12,"detail":"($self,@probs,$thresh)","signature":{"documentation":"1;\n# $Id: ProtMatrix.pm 16123 2009-09-17 12:57:27Z cjfields $\n#---------------------------------------------------------\n\n=head1 NAME\n\nBio::Matrix::PSM::ProtMatrix - SiteMatrixI implementation, holds a\nposition scoring matrix (or position weight matrix) with log-odds scoring\ninformation.\n\n=head1 SYNOPSIS\n\n   use Bio::Matrix::PSM::ProtMatrix;\n   # Create from memory by supplying probability matrix hash both as strings or\n   # arrays where the frequencies   Hash entries of the form lN refer to an array\n   # of position-specific log-odds scores for amino acid N. Hash entries of the\n   # form pN represent the position-specific probability of finding amino acid N.\n\n   my %param = (\n             'id' => 'A. thaliana protein atp1',\n             '-e_val' => $score,\n             'lS' => [ '-2', '3', '-3', '2', '-3', '1', '1', '3' ],\n             'lF' => [ '-1', '-4', '0', '-5', '0', '-5', '-4', '-4' ],\n             'lT' => [ '-1', '1', '0', '1', '-2', '-1', '0', '1' ],\n             'lN' => [ '-3', '-1', '-2', '3', '-5', '5', '-2', '0' ],\n             'lK' => [ '-2', '0', '-3', '2', '-3', '2', '-3', '-1' ],\n             'lY' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-4', '-4' ],\n             'lE' => [ '-3', '4', '-3', '2', '-4', '-2', '-3', '2' ],\n             'lV' => [ '0', '-2', '1', '-4', '1', '-4', '-1', '-3' ],\n             'lQ' => [ '-1', '0', '-2', '3', '-4', '1', '-3', '0' ],\n             'lM' => [ '8', '-3', '8', '-3', '1', '-3', '-3', '-3' ],\n             'lC' => [ '-2', '-3', '-3', '-4', '-3', '-4', '-3', '-3' ],\n             'lL' => [ '1', '-3', '1', '-4', '3', '-4', '-2', '-4' ],\n             'lA' => [ '-2', '1', '-2', '0', '-2', '-2', '2', '2' ],\n             'lW' => [ '-2', '-4', '-3', '-5', '-4', '-5', '-5', '-5' ],\n             'lP' => [ '-3', '-2', '-4', '-3', '-1', '-3', '6', '-3' ],\n             'lH' => [ '-2', '-2', '-3', '-2', '-5', '-2', '-2', '-3' ],\n             'lD' => [ '-4', '-1', '-3', '1', '-3', '-1', '-3', '4' ],\n             'lR' => [ '-2', '-1', '-3', '0', '-4', '4', '-4', '-3' ],\n             'lI' => [ '0', '-3', '0', '-4', '6', '-4', '-2', '-2' ],\n             'lG' => [ '-4', '-2', '-4', '-2', '-5', '-3', '-1', '-2' ],\n             'pS' => [ '0', '33', '0', '16', '1', '12', '11', '25' ],\n             'pF' => [ '0', '0', '2', '0', '3', '0', '0', '0' ],\n             'pT' => [ '0', '8', '7', '10', '1', '2', '7', '8' ],\n             'pN' => [ '0', '0', '2', '13', '0', '36', '1', '4' ],\n             'pK' => [ '0', '5', '0', '13', '1', '15', '0', '2' ],\n             'pY' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pE' => [ '0', '41', '1', '12', '0', '0', '0', '15' ],\n             'pV' => [ '0', '3', '9', '0', '2', '0', '3', '1' ],\n             'pQ' => [ '0', '0', '0', '15', '0', '4', '0', '3' ],\n             'pM' => [ '100', '0', '66', '0', '2', '0', '0', '0' ],\n             'pC' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pL' => [ '0', '0', '8', '0', '25', '0', '4', '0' ],\n             'pA' => [ '0', '10', '1', '9', '2', '0', '22', '16' ],\n             'pW' => [ '0', '0', '0', '0', '0', '0', '0', '0' ],\n             'pP' => [ '0', '0', '0', '0', '3', '1', '45', '0' ],\n             'pH' => [ '0', '0', '0', '0', '0', '0', '1', '0' ],\n             'pD' => [ '0', '0', '1', '7', '2', '2', '0', '22' ],\n             'pR' => [ '0', '0', '0', '3', '0', '27', '0', '0' ],\n             'pI' => [ '0', '0', '3', '0', '59', '1', '2', '3' ],\n             'pG' => [ '0', '0', '0', '1', '0', '0', '4', '1' ],\n   );\n\n   my $matrix = Bio::Matrix::PSM::ProtMatrix( %param );\n\n\n   my $site = Bio::Matrix::PSM::ProtMatrix->new(%param);\n   # Or get it from a file:\n   use Bio::Matrix::PSM::IO;\n   my $psmIO = Bio::Matrix::PSM::IO->new(-file => $file, -format => 'psi-blast');\n   while (my $psm = $psmIO->next_psm) {\n      #Now we have a Bio::Matrix::PSM::Psm object, \n      # see Bio::Matrix::PSM::PsmI for details\n      #This is a Bio::Matrix::PSM::ProtMatrix object now\n      my $matrix = $psm->matrix;   \n   }\n\n   # Get a simple consensus, where alphabet is:\n   # {A, R, N, D, C, Q, E, G, H, I, L, K, M, F, P, S, T, W, Y, V,}\n   # choosing the highest probability or N if prob is too low\n   my $consensus = $site->consensus;\n\n   # Retrieving and using regular expressions:\n   my $regexp = $site->regexp;\n   my $count = grep($regexp,$seq);\n   my $count = ($seq=~ s/$regexp/$1/eg);\n   print \"Motif $mid is present $count times in this sequence\\n\";\n\n=head1 DESCRIPTION\n\nProtMatrix is designed to provide some basic methods when working with\nposition scoring (weight) matrices related to protein sequences.  A\nprotein PSM consists of 20 vectors with 20 frequencies (one per amino\nacid per position).  This is the minimum information you should\nprovide to construct a PSM object.  The vectors can be provided as\nstrings with frequencies where the frequency is {0..a} and a=1. This\nis the way MEME compressed representation of a matrix and it is quite\nuseful when working with relational DB.  If arrays are provided as an\ninput (references to arrays actually) they can be any number, real or\ninteger (frequency or count).\n\nWhen creating the object the constructor will check for positions that\nequal 0.  If such is found it will increase the count for all\npositions by one and recalculate the frequency.  Potential bug - if\nyou are using frequencies and one of the positions is 0 it will change\nsignificantly.  However, you should never have frequency that equals\n0.\n\nThrows an exception if: You mix as an input array and string (for\nexample A matrix is given as array, C - as string).  The position\nvector is (0,0,0,0).  One of the probability vectors is shorter than\nthe rest.\n\nSummary of the methods I use most frequently (details bellow):\n\n   iupac - return IUPAC compliant consensus as a string\n   score - Returns the score as a real number\n   IC - information content. Returns a real number\n   id - identifier. Returns a string\n   accession - accession number. Returns a string\n   next_pos - return the sequence probably for each letter, IUPAC\n         symbol, IUPAC probability and simple sequence\n   consenus letter for this position. Rewind at the end. Returns a hash.\n   pos - current position get/set. Returns an integer.\n   regexp - construct a regular expression based on IUPAC consensus.\n         For example AGWV will be [Aa][Gg][AaTt][AaCcGg]\n   width - site width\n   get_string - gets the probability vector for a single base as a string.\n   get_array - gets the probability vector for a single base as an array.\n   get_logs_array - gets the log-odds vector for a single base as an array.\n\nNew methods, which might be of interest to anyone who wants to store\nPSM in a relational database without creating an entry for each\nposition is the ability to compress the PSM vector into a string with\nlosing usually less than 1% of the data.  this can be done with:\n\n   my $str=$matrix->get_compressed_freq('A');\nor\n\n   my $str=$matrix->get_compressed_logs('A');\n\nLoading from a database should be done with new, but is not yet implemented.\nHowever you can still uncompress such string with:\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1,1); for PSM\n\nor\n\n   my @arr=Bio::Matrix::PSM::_uncompress_string ($str,1000,2); for log odds\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to one\nof the Bioperl mailing lists.   Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nthe bugs and their resolution.  Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 AUTHOR - James Thompson\n\nEmail tex@biosysadmin.com\n\n=head1 APPENDIX\n\n\n\n# Let the code begin...\npackage Bio::Matrix::PSM::ProtMatrix;\nuse strict;\n\nuse base qw(Bio::Root::Root Bio::Matrix::PSM::SiteMatrixI);\n\n=head2 new\n\n Title    : new\n Usage    : my $site = Bio::Matrix::PSM::ProtMatrix->new( \n               %probs,\n               %logs,\n               -IC    => $ic,\n               -e_val => $score, \n               -id    => $mid\n               -model => \\%model\n            );\n Function : Creates a new Bio::Matrix::PSM::ProtMatrix object from memory\n Throws   : If inconsistent data for all vectors (all 20 amino acids) is\n               provided, if you mix input types (string vs array) or if a\n               position freq is 0.\n Example  :\n Returns  : Bio::Matrix::PSM::ProtMatrix object\n Args     : Hash references to log-odds scores and probabilities for\n            position-specific scoring info, e-value (optional), information\n            content (optional), id (optional), model for background distribution\n            of proteins (optional).\n\n\nsub new {\n   my ($class, @args) = @_;\n   my $self = $class->SUPER::new(@args);\n   my $consensus;\n   #Too many things to rearrange, and I am creating simultanuously >500 \n   # such objects routinely, so this becomes performance issue\n   my %input;\n   while( @args ) {\n     (my $key = shift @args) =~ s/-//gi; #deletes all dashes (only dashes)!\n     $input{$key} = shift @args;\n   }\n\n   # get a protein alphabet for processing log-odds scores and probabilities\n   # maybe change this later on to allow for non-standard aa lists? \n   my @alphabet = qw/A  R  N  D  C  Q  E  G  H  I  L  K  M  F  P  S  T  W  Y  V/;\n\n   foreach my $aa (@alphabet) {\n      $self->{\"log$aa\"}  = defined($input{\"l$aa\"}) ? $input{\"l$aa\"} \n         : $self->throw(\"Error: No log-odds information for $aa!\");\n      $self->{\"prob$aa\"}  = defined($input{\"p$aa\"}) ? $input{\"p$aa\"} \n         : $self->throw(\"Error: No probability information for $aa!\");\n   }\n   \n   $self->{_position} = 0;\n   $self->{IC}        = $input{IC};\n   $self->{e_val}     = $input{e_val};\n   $self->{sites}     = $input{sites};\n   $self->{width}     = $input{width};\n   $self->{accession_number} = $input{accession_number};\n   $self->{_correction} = defined($input{correction}) ?\n      $input{correction} : 1 ; # Correction might be unwanted- supply your own\n   # No id provided, null for the sake of rel db\n   $self->{id} = defined($input{id}) ? $input{id} : 'null'; \n   $self->{_alphabet} = \\@alphabet;\n\n   #Make consensus, throw if any one of the vectors is shorter\n   $self = _calculate_consensus($self,$input{model});\n   return $self;\n}\n\n=head2 alphabet\n\n Title    : Returns an array (or array reference if desired) to the alphabet \n Usage    :\n Function : Returns an array (or array reference) containing all of the\n            allowable characters for this matrix.\n Throws   :\n Example  :\n Returns  : Array or arrary reference.\n Args     :\n\n\nsub alphabet {\n   my $self = shift;\n   if ( wantarray ) {\n      return $self->{_alphabet};\n   } else {\n      return @{$self->{_alphabet}};\n   }\n}\n\n=head2 _calculate_consensus\n\n Title    : _calculate_consensus\n Usage    :\n Function : Calculates the consensus sequence for this matrix. \n Throws   :\n Example  :\n Returns  :\n Args     :\n\n\nsub _calculate_consensus {\n   my $self   = shift;\n   my $thresh = shift;\n   \n   # verify that all of the array lengths in %probs are the same\n   my @lengths = map { scalar(@$_) } map {$self->{\"prob$_\"}} @{ $self->{_alphabet} };\n   my $len = shift @lengths;\n   for ( @lengths ) {\n      if ( $_ ne $len ) { $self->throw( \"Probability matrix is damaged!\\n\" ) };\n   }\n\n   # iterate over probs, generate the most likely sequence and put it into \n   # $self->{seq}. Put the probability of this sequence into $self->{seqp}.\n   for ( my $i = 0; $i < $len; $i++ ) {\n      # get a list of all the probabilities at position $i, ordered by $self->{_alphabet}\n      my @probs = map { ${$self->{\"prob$_\"}}[$i] } @{ $self->{_alphabet} };\n      # calculate the consensus of @probs, put sequence into seqp and probabilities into seqp\n      (${$self->{seq}}[$i],${$self->{seqp}}[$i]) = $self->_to_cons( @probs, $thresh );\n   }\n\n   return $self;\n}\n\n=head2 next_pos\n\n Title    : next_pos\n Usage    :\n Function : Retrives the next position features: frequencies for all 20 amino\n            acids, log-odds scores for all 20 amino acids at this position,\n            the main (consensus) letter at this position, the probability\n            for the consensus letter to occur at this position and the relative\n            current position as an integer.\n Throws   :\n Example  :\n Returns  : hash (or hash reference) (pA,pR,pN,pD,...,logA,logR,logN,logD,aa,prob,rel)\n            - pN entries represent the probability for amino acid N\n            to be at this position\n            - logN entries represent the log-odds score for having amino acid\n            N at this position\n            - aa is the consensus amino acid\n            - prob is the probability for the consensus amino acid to be at this \n            position\n            - rel is the relative index of the current position (integer)\n Args      : none\n\n\n\nsub next_pos {\n      my $self = shift;\n         $self->throw(\"instance method called on class\") unless ref $self;\n\n      my $len = @{$self->{seq}};\n      my $pos = $self->{_position};\n\n      # return a PSM if we're still within range\n      if ($pos<$len) {\n\n         my %probs = map { (\"p$_\", ${$self->{\"prob$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my %logs  = map { (\"l$_\", ${$self->{\"log$_\"}}[$pos]) } @{$self->{_alphabet}};\n         my $base  = ${$self->{seq}}[$pos];\n         my $prob  = ${$self->{seqp}}[$pos];\n\n         $self->{_position}++;\n         my %hash = ( %probs, %logs, base => $base, rel => $pos, prob => $prob );\n         \n         # decide whether to return the hash or a reference to it\n         if ( wantarray ) {\n            return %hash;\n         } else {\n            return \\%hash;\n         }\n   } else { # otherwise, reset $self->{_position} and return nothing\n      $self->{_position} = 0; \n      return;\n   }\n}\n\n\n=head2 curpos\n\n Title    : curpos\n Usage    :\n Function : Gets/sets the current position. \n Throws   :\n Example  :\n Returns  : Current position (integer).\n Args     : New position (integer).\n\n\nsub curpos {\n      my $self = shift;\n      if (@_) { $self->{_position} = shift; }\n      return $self->{_position};\n}\n\n\n=head2 e_val\n\n Title    : e_val\n Usage    :\n Function : Gets/sets the e-value\n Throws   :\n Example  :\n Returns  : \n Args     : real number\n\n\nsub e_val {\n      my $self = shift;\n      if (@_) { $self->{e_val} = shift; }\n      return $self->{e_val};\n}\n\n\n=head2 IC\n\n Title    : IC\n Usage    :\n Function : Position-specific information content.\n Throws   :\n Example  :\n Returns  : Information content for current position.\n Args     : Information content for current position.\n\n\nsub IC {\n      my $self = shift;\n      if (@_) { $self->{IC} = shift; }\n      return $self->{IC};\n}\n\n=head2 accession_number\n\n Title    : accession_number\n Usage    :\n Function: accession number, this will be unique id for the ProtMatrix object as\n            well for any other object, inheriting from ProtMatrix.\n Throws   :\n Example  :\n Returns  : New accession number (string)\n Args     : Accession number (string)\n\n\nsub accession_number {\n      my $self = shift;\n      if (@_) { $self->{accession_number} = shift; }\n      return $self->{accession_number};\n}\n\n=head2 consensus\n\n Title    : consensus\n Usage    :\n Function : Returns the consensus sequence for this PSM.\n Throws   : if supplied with thresold outisde 5..10 range\n Example  :\n Returns  : string\n Args     : (optional) threshold value 5 to 10 (corresponds to 50-100% at each position\n\n\nsub consensus {\n   my $self = shift;\n   my $thresh=shift;\n   $self->_calculate_consensus($thresh) if ($thresh); #Change of threshold\n   my $consensus='';\n\n   foreach my $letter (@{$self->{seq}}) {\n       $consensus .= $letter;\n   }\n\n   return $consensus;\n}\n\nsub IUPAC {\n   my $self = shift;\n   return $self->consensus;\n}\n\n\n=head2 get_string\n\n Title   : get_string\n Usage   :\n Function: Returns given probability vector as a string. Useful if you want to\n            store things in a rel database, where arrays are not first choice\n Throws  : If the argument is outside {A,C,G,T}\n Example :\n Returns : string\n Args    : character {A,C,G,T}\n\n\nsub get_string {\n   my $self = shift;\n   my $base = shift;\n   my $string = '';\n\n   my @prob = @{$self->{\"prob$base\"}};\n   if ( ! @prob ) {\n      $self->throw( \"No such base: $base\\n\");\n   }\n\n   foreach my $prob (@prob) {\n      my $corrected = $prob*10;\n      my $next = sprintf(\"%.0f\",$corrected);\n      $next = 'a' if ($next eq '10');\n      $string .= $next;\n   }\n   return $string;\n}\n\n\n\n=head2 width\n\n Title    : width\n Usage    :\n Function : Returns the length of the site\n Throws   :\n Example  :\n Returns  : number\n Args     :\n\n\nsub width {\n   my $self = shift;\n   my $width = @{$self->{probA}};\n   return $width;\n}\n\n=head2 get_array\n\n Title    : get_array\n Usage    :\n Function : Returns an array with frequencies for a specified amino acid.\n Throws   :\n Example  :\n Returns  : Array representing frequencies for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"prob$letter\"}}; \n}\n\n\n=head2 get_logs_array\n\n Title    : get_logs_array\n Usage    :\n Function : Returns an array with log_odds for a specified base\n Throws   :\n Example  :\n Returns  : Array representing log-odds scores for specified amino acid.\n Args     : Single amino acid (character).\n\n\nsub get_logs_array {\n   my $self = shift;\n   my $letter = uc(shift);\n\n   $self->throw (\"No such base: $letter!\\n\") unless grep { /$letter/ } @{$self->{_alphabet}};\n\n   return @{$self->{\"log$letter\"}}; \n}\n\n=head2 id\n\n Title    : id\n Usage    :\n Function : Gets/sets the site id\n Throws   :\n Example  :\n Returns  : string\n Args     : string\n\n\nsub id {\n      my $self = shift;\n      if (@_) { $self->{id} = shift; }\n      return $self->{id};\n}\n\n=head2 regexp\n\n Title    : regexp\n Usage    :\n Function : Returns a case-insensitive regular expression which matches the\n            IUPAC convention.  X's in consensus sequence will match anything.     \n Throws   :\n Example  :\n Returns  : string\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n\n\nsub regexp {\n   my $self = shift;\n   my $threshold = 20;\n   if ( @_ ) { my $threshold = shift };\n\n   my @alphabet = @{$self->{_alphabet}};\n   my $width = $self->width;\n   my (@regexp, $i);\n   for ( $i = 0; $i < $width; $i++ ) {\n      # get an array of the residues at this position with p > $threshold\n      my @letters = map { uc($_).lc($_) } grep { $self->{\"prob$_\"}->[$i] >= $threshold } @alphabet;\n\n      my $reg;\n      if ( scalar(@letters) == 0 ) {\n         $reg = '\\.';\n      } else {\n         $reg = '['.join('',@letters).']';\n      }\n      push @regexp, $reg;\n   }\n\n   if ( wantarray ) { \n      return @regexp;\n   } else {\n      return join '', @regexp;\n   }\n}\n\n\n=head2 regexp_array\n\n Title    : regexp_array\n Usage    :\n Function : Returns an array of position-specific regular expressions.\n             X's in consensus sequence will match anything.      \n Throws   :\n Example  :\n Returns  : Array of position-specific regular expressions.\n Args     : Threshold for calculating consensus sequence (number in range 0-100\n            representing a percentage). Threshold defaults to 20.\n Notes    : Simply calls regexp method in list context.\n\n\nsub regexp_array {\n   my $self = shift;\n   \n   return @{ $self->regexp };\n}\n\n\n=head2 _compress_array\n\n Title    : _compress_array\n Usage    :\n Function :  Will compress an array of real signed numbers to a string (ie vector of bytes)\n             -127 to +127 for bi-directional(signed) and 0..255 for unsigned ;\n Throws   :\n Example  :  Internal stuff\n Returns  :  String\n Args     :  array reference, followed by max value and direction (optional, defaults to 1),\n             direction of 1 is unsigned, anything else is signed. \n\n\nsub _compress_array {\n   my ($array,$lm,$direct)=@_;\n   my $str;\n   return unless(($array) && ($lm));\n   $direct=1 unless ($direct);\n   my $k1= ($direct==1) ? (255/$lm) : (127/$lm);\n   foreach my $c (@{$array}) {\n      $c=$lm if ($c>$lm);\n      $c=-$lm if (($c<-$lm) && ($direct !=1));\n      $c=0 if (($c<0) && ($direct ==1));\n      my $byte=int($k1*$c);\n      $byte=127+$byte if ($direct !=1);#Clumsy, should be really shift the bits\n      my $char=chr($byte);\n      $str.=$char;\n   }\n   return $str;\n}\n\n=head2 _uncompress_string\n\n Title    : _uncompress_string\n Usage    :\n Function :   Will uncompress a string (vector of bytes) to create an array of real\n                  signed numbers (opposite to_compress_array)\n Throws   :\n Example  :   Internal stuff\n Returns  :   string, followed by max value and direction (optional, defaults to 1),\n              direction of 1 is unsigned, anything else is signed.\n Args     :   array\n\n\nsub _uncompress_string {\n   my ($str,$lm,$direct)=@_;\n   my @array;\n   return unless(($str) && ($lm));\n   $direct=1 unless ($direct);\n   my $k1= ($direct==1) ? (255/$lm) : (127/$lm);\n   while (my $c=chop($str)) {\n      my $byte=ord($c);\n      $byte=$byte-127 if ($direct !=1);#Clumsy, should be really shift the bits\n      my $num=$byte/$k1;\n      unshift @array,$num;\n   }\n\n   return @array;\n}\n\n=head2 get_compressed_freq\n\n Title    : get_compressed_freq\n Usage    :\n Function:   A method to provide a compressed frequency vector. It uses one byte to\n             code the frequence for one of the probability vectors for one position.\n             Useful for relational database. Improvment of the previous 0..a coding.\n Throws   :\n Example  :   my $strA=$self->get_compressed_freq('A');\n Returns  :   String\n Args     :   char \n\n\nsub get_compressed_freq {\n   my $self=shift;\n   my $base=shift;\n   my $string='';\n   my @prob;\n   BASE: {\n      if ($base eq 'A') {\n         @prob = @{$self->{probA}} unless (!defined($self->{probA}));\n         last BASE;\n      }\n         if ($base eq 'G') {\n         @prob = @{$self->{probG}} unless (!defined($self->{probG}));\n         last BASE;\n      }\n         if ($base eq 'C') {\n         @prob = @{$self->{probC}} unless (!defined($self->{probC}));\n         last BASE;\n      }\n         if ($base eq 'T') {\n         @prob = @{$self->{probT}} unless (!defined($self->{probT}));\n         last BASE;\n      }\n      $self->throw (\"No such base: $base!\\n\");\n   }\n   my $str= _compress_array(\\@prob,1,1);\n   return $str;\n}\n\n=head2 sequence_match_weight\n\n Title    : sequence_match_weight\n Usage    :\n Function :   This method will calculate the score of a match, based on the PSM\n              if such is associated with the matrix object. Returns undef if no\n              PSM data is available.\n Throws   :   if the length of the sequence is different from the matrix width\n Example  :   my $score=$matrix->sequence_match_weight('ACGGATAG');\n Returns  :   Floating point\n Args     :   string\n\n\nsub sequence_match_weight {\n   my ($self,$seq)=@_;\n   return unless ($self->{logA});\n\n   my $seqlen = length($seq);\n   my $width  = $self->width;\n   $self->throw(\"Error: Input sequence size ($seqlen) not equal to PSM size ($width)!\\n\")\n      unless (length($seq) == $self->width);\n\n   my ($score,$i) = (0,0);\n   foreach my $letter ( split //, $seq ) {\n      # add up the score for this position\n      $score += $self->{\"log$letter\"}->[$i];\n      $i++;\n   }\n   return $score;\n}\n\n\n=head2 _to_IUPAC\n\n Title   : _to_IUPAC\n Usage   :\n Function: Converts a single position to IUPAC compliant symbol and returns its probability.\n            Currently returns the most likely amino acid/probability combination.\n Throws  :\n Example :\n Returns : char, real number representing an amino acid and a probability.\n Args    : real numbers for all 20 amino acids (ordered by alphabet contained\n            in $self->{_alphabet}, minimum probability threshold.","parameters":[{"label":"$self"},{"label":"@probs"},{"label":"$thresh"}],"label":"_to_IUPAC($self,@probs,$thresh)"},"containerName":"main::","definition":"sub"},{"line":814,"kind":12,"name":"_alphabet"},{"kind":13,"line":815,"containerName":null,"name":"$prob","definition":"my","localvar":"my"},{"containerName":null,"name":"@probs","kind":13,"line":815},{"kind":13,"line":816,"containerName":null,"name":"$prob"},{"containerName":null,"name":"%max_prob","line":816,"kind":13},{"name":"$IUPAC_aa","containerName":null,"kind":13,"line":817},{"kind":13,"line":817,"name":"$aa","containerName":null},{"line":818,"kind":13,"name":"$max_prob","containerName":null},{"kind":13,"line":818,"name":"$prob","containerName":null},{"name":"$IUPAC_aa","containerName":null,"line":822,"kind":13},{"line":822,"kind":13,"name":"$max_prob","containerName":null},{"line":838,"children":[],"kind":12,"range":{"end":{"character":9999,"line":840},"start":{"line":838,"character":0}},"containerName":"main::","definition":"sub","name":"_to_cons"}],"version":5}